package com.contentsquare.android.internal.util;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;
import android.view.ViewGroup;

import com.contentsquare.android.internal.DeviceInfo;
import com.contentsquare.android.internal.Session;
import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.network.NetworkTransaction;
import com.contentsquare.android.internal.screengraph.Screengraph;
import com.contentsquare.android.internal.screengraph.ScreenshotGenerator;

import org.json.JSONObject;

/**
 * Temporary utility (yeah right), to send screengraphs in god mode.
 * If we start sending in "non-god" mode, with real clients, we should migrate this code to a proper
 * sending mechanism with save to disk fallbacks and similar mechanisms.
 * Note : this class is a "temp class containing debug only logic creating and sending screengraphs.
 */
public class ScreenRecorder {
    private final Logger mLogger = new Logger("ScreenRecorder");
    @NonNull
    private final UriBuilder mUriBuilder;
    @NonNull
    private final DeviceInfo mDeviceInfo;
    @NonNull
    private final Session mSession;
    @NonNull
    private final ViewUtil mViewUtil;


    /**
     * Builds an instance of the {@link ScreenRecorder}.
     *
     * @param deviceInfo as {@link DeviceInfo}
     * @param session    as {@link Session}
     * @param uriBuilder as {@link UriBuilder}
     * @param viewUtil   as {@link ViewUtil}
     */
    public ScreenRecorder(
            @NonNull DeviceInfo deviceInfo,
            @NonNull Session session,
            @NonNull UriBuilder uriBuilder,
            @NonNull ViewUtil viewUtil) {
        mDeviceInfo = deviceInfo;
        mSession = session;
        mUriBuilder = uriBuilder;
        mViewUtil = viewUtil;
    }

    /**
     * Captures the current screen and sends the info to our servers.
     * The captures are two in type.
     * A {@link JSONObject} representing the screengraph.
     * A serialized bitmap object.
     * Each of them is being sent in it's on batch to different endpoints.
     *  @param root         the root view we're capturing from. Usually the decorView.
     * @param targetPathId as {@code int} - the hashcode of the target path
     */
    public void capture(@Nullable ViewGroup root, @Nullable String url, int targetPathId) {
        if (root == null) {
            mLogger.w("Failed to capture screen, decorView is null");
            return;
        }
        if (mSession.canTakeScreenShots()) {
            createAndSendScreengraph(root, url, targetPathId);
            createAndSendScreenshot(root, url, targetPathId);
        } else {
            mLogger.i("Skipping screen capture, disabled in config (either project or god).");
        }

    }

    /**
     * Creates and sends a screengraph from the provided view group.
     *  @param root         the ViewGroup to capture
     * @param url          as {@link String} - the url for the current ViewGroup
     * @param targetPathId as {@code int} - the hashcode of the target path
     */
    @VisibleForTesting
    void createAndSendScreengraph(@NonNull ViewGroup root,
                                  @Nullable String url,
                                  int targetPathId) {
        JSONObject screen = getScreengraph().obtain(root, url, targetPathId);

        final String endpoint = mUriBuilder.buildScreengraphUrl(
                mSession.getGodModeConfig().getScreengraphUploadUrl(),
                mSession.getRunConfiguration().getCsProjectId(),
                mDeviceInfo.getDeviceIntType());

        sendJsonToServer(screen, endpoint);
    }

    /**
     * Creates and sends a json containing the serialized screenshot of the provided ViewGroup.
     *  @param root         the ViewGroup to capture
     * @param targetPathId as {@code int} - the hashcode of the target path
     */
    @VisibleForTesting
    void createAndSendScreenshot(@NonNull ViewGroup root, @Nullable String url, int targetPathId) {
        JSONObject screenshotJson = getScreenshotGenerator().obtain(url, root, targetPathId);
        final String endpoint = mUriBuilder.buildScreenshotUrl(
                mSession.getGodModeConfig().getScreengraphUploadUrl(),
                mSession.getRunConfiguration().getCsProjectId(),
                mDeviceInfo.getDeviceIntType());

        sendJsonToServer(screenshotJson, endpoint);
    }

    @VisibleForTesting
    Screengraph getScreengraph() {
        return new Screengraph();
    }

    @VisibleForTesting
    ScreenshotGenerator getScreenshotGenerator() {
        return new ScreenshotGenerator(mViewUtil);
    }

    /**
     * Sends screengraphs to predefined url.
     * Note: This is a debug level method, please consider it as volatile.
     *
     * @param screen - the screen to be sent.
     */
    @VisibleForTesting
    void sendJsonToServer(@Nullable JSONObject screen, @NonNull final String destination) {
        if (screen == null) {
            // nothing to send
            mLogger.d("Generated Screengraph is null, aborting...");
            return;
        }

        NetworkTransaction.NetworkTransactionListener listener =
                new NetworkTransaction.NetworkTransactionListener() {

                    @Override
                    public void pass() {
                        mLogger.d("Successfully sent to %s", destination);
                    }

                    @Override
                    public void fail() {
                        mLogger.d("Failed sending to %s", destination);
                    }
                };

        NetworkTransaction nt = new NetworkTransaction();
        nt.scheduleSend(screen, destination, listener);
    }
}