package com.contentsquare.android.internal.util;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.contentsquare.android.internal.logging.Logger;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;

/**
 * Static utility methods pertaining to {@code String} or {@code CharSequence} instances.
 * Originally from Guava, copied so that we don't get the dependency, as it's quite expensive.
 */
public class Strings {

    /**
     * Constant String - UTF-8 charset. Use this as a static import whereever possible.
     */
    static final String UTF_8 = "UTF-8";

    @NonNull
    private static final Logger LOGGER = new Logger("Strings");

    /**
     * Prevent object instantiations.
     */
    private Strings() {
    }

    /**
     * Returns the given string if it is non-null; the empty string otherwise.
     *
     * @param string the string to test and possibly return
     * @return {@code string} itself if it is non-null; {@code ""} if it is null
     */
    @NonNull
    public static String nullToEmpty(@Nullable String string) {
        return (string == null) ? "" : string;
    }

    /**
     * Returns the given string if it is nonempty; {@code null} otherwise.
     *
     * @param string the string to test and possibly return
     * @return {@code string} itself if it is nonempty; {@code null} if it is empty or null
     */
    @Nullable
    public static String emptyToNull(@Nullable String string) {
        return isNullOrEmpty(string) ? null : string;
    }

    /**
     * Returns {@code true} if the given string is null or is the empty string.
     * Consider normalizing your string references with {@link #nullToEmpty}. If you do, you can
     * use {@link String#isEmpty()} instead of this method, and you won't need special null-safe
     * forms of methods like {@link String#toUpperCase} either. Or, if you'd like to normalize "in
     * the other direction," converting empty strings to {@code null},
     * you can use {@link #emptyToNull}.
     *
     * @param string a string reference to check
     * @return {@code true} if the string is null or is the empty string
     */
    public static boolean isNullOrEmpty(@Nullable String string) {
        return string == null || string.isEmpty();
    }

    /**
     * Encodes a String in URL ready format using {@link java.net.URLEncoder} in UTF-8.
     *
     * @param string the string to encode.
     * @return a URLencoded representation of this string.
     */
    public static String encode(@Nullable String string) {
        if (isNullOrEmpty(string)) {
            return string;
        } else {
            try {
                return URLEncoder.encode(string, UTF_8);
            } catch (UnsupportedEncodingException e) {
                LOGGER.w(e, "Failed to encode sting to UTF, exception : ");
                return null;
            }
        }
    }
}
