package com.contentsquare.android.internal.util;

import android.support.annotation.NonNull;

import com.contentsquare.android.internal.DeviceInfo;

import java.util.Locale;

/**
 * Builds URL's for the base or project configuration files,
 * depending on debug state of the session.
 * example: https://mobile-dev.content-square.net/iOS/config/Base.json
 * example: https://mobile-production.content-square.net/iOS/config/mhacks.json
 */
public class UriBuilder {
    /**
     * The Endpoint for Project Json Configuration when the application used for production.
     */
    public static final String PROJECT_CONFIG_RELEASE_URL =
            "https://mobile-production.content-square.net/android/config/v1/";

    /**
     * The Endpoint for Project Json Configuration when the application used for debug mode.
     */
    public static final String PROJECT_CONFIG_DEBUG_URL =
            "https://mobile-dev.content-square.net/android/config/v1/";

    /**
     * The default Endpoint for Project Json GodMode Configuration.
     */
    public static final String PROJECT_GOD_MODE_CONFIG_URL =
            "https://s3-eu-west-1.amazonaws.com/cs-mobile-dev/android/god/cs-override.json";

    private static final String EXTENSION = ".json";
    private static final String EVENTS_VERSION = "v2";
    private static final String EVENTS_ENDPOINT = "events";

    private static final String SCREENGRAPH_VERSION = "v1";

    @NonNull
    private final DeviceInfo mDeviceInfo;

    /**
     * Constructor for a config url builder, internal dependencies are injected by dagger.
     *
     * @param deviceInfo as {@link DeviceInfo}
     */
    public UriBuilder(@NonNull DeviceInfo deviceInfo) {

        this.mDeviceInfo = deviceInfo;
    }

    /**
     * Builds a Endpoint URL to retrieve the
     * {@link com.contentsquare.android.internal.config.ProjectConfiguration} from server side.
     *
     * @param projectId the id of the project.
     * @return a url pointing to the specified project's configuration file.
     */
    @NonNull
    public String buildProjectConfigUrl(@NonNull String projectId) {
        return buildConfigUrl(projectId, mDeviceInfo.isDebugBuild());
    }

    /**
     * Builds a Debug Endpoint URL to retrieve the
     * {@link com.contentsquare.android.internal.config.ProjectConfiguration} from server side.
     *
     * @param projectId the id of the project.
     * @return a url pointing to the specified project's configuration file.
     */
    @NonNull
    public String buildProjectConfigUrlForDebug(@NonNull String projectId) {
        return buildConfigUrl(projectId, true);
    }

    @NonNull
    private String buildConfigUrl(final String projectId, boolean isDebug) {
        return configEndpoint(isDebug)
                + projectId.toLowerCase(Locale.getDefault())
                + EXTENSION;
    }

    @NonNull
    private String configEndpoint(boolean isDebug) {
        return isDebug ? PROJECT_CONFIG_DEBUG_URL : PROJECT_CONFIG_RELEASE_URL;
    }

    /**
     * Provides the Endpoint URL for the God Mode Config Json file.
     *
     * @return the url as {@link String}
     */
    @NonNull
    public String godModeConfigEndpoint() {
        return PROJECT_GOD_MODE_CONFIG_URL;
    }

    /**
     * Builds a URL endpoint for the server, depending on host defined in the configuration.
     *
     * @param servername the server
     * @return a url
     */
    @NonNull
    public String buildEventsUrl(@NonNull String servername) {

        return servername + "/mobile/" + EVENTS_VERSION + "/" + EVENTS_ENDPOINT;
    }

    /**
     * Returns a built URL where to send screengraphs.
     *
     * @param servername the server
     * @param projectId  the id of the current project (to be retreived from the configuration)
     * @param deviceType the type of the device (1: mobile, 2: tablet)
     * @return a url
     */
    @NonNull
    String buildScreengraphUrl(@NonNull String servername, int projectId, int deviceType) {

        return servername
                + "/screengraphs/"
                + SCREENGRAPH_VERSION
                + "/screengraph/project/"
                + projectId
                + "/device/"
                + deviceType;
    }

    /**
     * Returns a built URL where to send screenshots.
     *
     * @param servername the server
     * @param projectId  the id of the current project (to be retreived from the configuration)
     * @param deviceType the type of the device (1: mobile, 2: tablet)
     * @return a url
     */
    @NonNull
    String buildScreenshotUrl(@NonNull String servername, int projectId, int deviceType) {

        return servername
                + "/screengraphs/"
                + SCREENGRAPH_VERSION
                + "/screenshot/project/"
                + projectId
                + "/device/"
                + deviceType;
    }
}
