package com.contentsquare.android.internal.util;

import static android.util.Base64.DEFAULT;
import static com.contentsquare.android.internal.util.ResourceUtils.NULL_STRING_ID;
import static com.contentsquare.android.internal.util.ResourceUtils.getResourceEntryName;

import android.content.Context;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.Drawable;
import android.support.annotation.NonNull;
import android.support.v4.util.Pair;
import android.util.Base64;
import android.util.DisplayMetrics;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AdapterView;

import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.screengraph.JsonMetadataView;
import com.contentsquare.android.internal.screengraph.JsonView;
import com.contentsquare.android.internal.ui.processor.descriptors.PathDescriptor;

import java.io.ByteArrayOutputStream;
import java.util.Comparator;
import java.util.PriorityQueue;
import java.util.Queue;
import java.util.concurrent.ArrayBlockingQueue;

import hugo.weaving.DebugLog;

/**
 * Utilities related to traversing, manipulating or processing views and view Hierarchies.
 */
public class ViewUtil {

    private static final String COLOR_TRANSPARENT = "#00FFFFFF";
    private static final int FAKE_VALUE = 0;
    private static final int COMPRESS_QUALITY = 100;
    private static final int COLOR_FORMAT = 0xFFFFFF;
    private final Logger mLogger = new Logger("ViewUtil");

    private static boolean viewSupportsAdapter(View biggest) {
        return (biggest instanceof AdapterView)
                || (biggest.getClass().toString().contains("RecyclerView"));
    }

    /**
     * This method converts dp unit to equivalent pixels, depending on device density.
     *
     * @param dps     A value in dp (density independent pixels) unit. Which we need to convert
     *                into pixels
     * @param context Context to get resources and device specific display metrics
     * @return A float value to represent px equivalent to dp depending on device density
     */
    public static int convertDpsToPixel(int dps, Context context) {
        final Resources resources = context.getResources();
        final DisplayMetrics metrics = resources.getDisplayMetrics();
        final float scale = ((float) metrics.densityDpi / DisplayMetrics.DENSITY_DEFAULT);
        return MathUtil.safeFloatToInt(dps * scale);
    }

    /**
     * This method converts device specific pixels to density independent pixels.
     *
     * @param px      A value in px (pixels) unit. Which we need to convert into db
     * @param context Context to get resources and device specific display metrics
     * @return A float value to represent dp equivalent to px value
     */
    public static int convertPixelsToDps(int px, Context context) {
        final Resources resources = context.getResources();
        final DisplayMetrics metrics = resources.getDisplayMetrics();
        final float scale = ((float) metrics.densityDpi / DisplayMetrics.DENSITY_DEFAULT);
        return MathUtil.safeFloatToInt(px / scale);
    }

    /**
     * Returns a {@see View} which is the biggest view in the hierarchy provided by the root.
     * The biggest, in reference to the most screen size taken.
     *
     * @param root the root of a view hierarchy
     * @return the view which has the most real-estate on the screen.
     */
    @SuppressWarnings("squid:S3776")
    // skip sonar complexity warning, this is complex by design...
    @DebugLog
    @NonNull
    public View getBiggestViewInHierarchy(@NonNull ViewGroup root) {
        // use a queue as a recursion substitute, the queue will always have a size of 1 as it will
        // only process a single view at a time.
        Queue<ViewGroup> queue = new ArrayBlockingQueue<>(1);
        // create a priority queue for pairs of views and their areas, the queue will sort them
        // and have the biggest on the top. The comparator will serve as a "auto-sorter"
        final PriorityQueue<Pair<View, Integer>> areas = new PriorityQueue<>(10,
                new Comparator<Pair<View, Integer>>() {
                    @Override
                    public int compare(Pair<View, Integer> one, Pair<View, Integer> two) {
                        return MathUtil.compareInts(one.second, two.second);
                    }
                });
        // push the root on the queue
        queue.add(root);
        mLogger.d("Finding the biggest segment in %s", root.toString());
        while (!queue.isEmpty()) {
            areas.clear();
            // get the element from the queue and remove it
            ViewGroup element = queue.poll();
            int childCount = element.getChildCount();
            if (childCount == 0) {
                // if the group has no children, than it's the biggest element.
                mLogger.d("View Group without children detected, returning", element.toString());
                return element;
            }

            // go through all children and find the biggest one.
            for (int i = 0; i < childCount; i++) {
                View child = element.getChildAt(i);
                if (child == null || child.getVisibility() != View.VISIBLE) {
                    //just in case child is ever null or invisible
                    mLogger.e("Child  was null or invisible, skipping, %s", child);
                    continue;
                }
                int area = MathUtil.calculateArea(child.getWidth(), child.getHeight());
                // add to a priority queue
                areas.add(new Pair<>(child, area));
            }
            if (areas.isEmpty()) {
                // either all children were invisible or were null, for someweird reason,
                // we return the parent which was the biggest one.
                return element;
            }
            // get the view with the biggest area, by just pooling from the priority queue,
            // the queue should have done the sorting for us, so the biggest one is on top.
            View biggest = areas.poll().first;
            if (viewSupportsAdapter(biggest)) {
                return biggest;
            } else if (biggest instanceof ViewGroup) {
                mLogger.d("adding child for processing : %s", biggest);
                queue.add((ViewGroup) biggest);
            } else {
                final View result = (View) biggest.getParent();
                mLogger.d("found biggest child, returning it's parent : %s", result);
                return result;
            }
        }
        // this will never happen. Due to the way this function is defined the IDE cannot see that
        // it will finish before this statement as all cases were covered.
        mLogger.e("we have an error in processing, returning the root view.");
        return root;
    }

    /**
     * Encodes the byte array into base64 string.
     *
     * @param imageByteArray - byte array
     * @return String a {@link String}
     */
    @NonNull
    public String encodeImage(@NonNull byte[] imageByteArray) {
        return Base64.encodeToString(imageByteArray, DEFAULT);
    }

    /**
     * Serializes this {@link View} to a {@link JsonView}.
     *
     * @param view a View node containing no child
     * @return a {@link JsonView} object with View data
     */
    @SuppressWarnings( {"squid:S1166", "squid:S1450"})
    public JsonView toObject(@NonNull View view) {
        final JsonView viewObj = new JsonView(this);
        int[] locationXy = new int[2];
        view.getLocationOnScreen(locationXy);
        PathDescriptor pathDecriptor = new PathDescriptor();


        JsonMetadataView metaObj = new JsonMetadataView();
        metaObj.setClassName(view.getClass().getSimpleName());
        metaObj.setFullPath(pathDecriptor.generateAnalyticsPath(view));
        metaObj.setChildOrder(FAKE_VALUE); //TODO: not used for now, will be implemented later

        viewObj.setHeight(view.getHeight());
        viewObj.setWidth(view.getWidth());
        viewObj.setPosX(locationXy[0]);
        viewObj.setPosY(locationXy[1]);
        viewObj.setVisible(view.getVisibility() == View.VISIBLE);
        viewObj.setMetadata(metaObj.toJson());

        if (view instanceof ViewGroup) {
            viewObj.setBackground(getBackgroundColor(view));
        } else {
            viewObj.setBitmap(toByteArray(view));
        }

        String descriptor = getResourceEntryName(view, NULL_STRING_ID);
        viewObj.setId(descriptor);

        return viewObj;
    }

    /**
     * Converts the supplied view into bitmap that will then be converted into a byte array.
     */
    @NonNull
    public byte[] toByteArray(@NonNull View view) {
        if ((view.getHeight() > 0) && (view.getWidth() > 0)) {
            Bitmap bitmap = Bitmap.createBitmap(view.getWidth(), view.getHeight(),
                    Bitmap.Config.ARGB_8888);
            Canvas canvas = new Canvas(bitmap);
            view.layout(view.getLeft(), view.getTop(), view.getRight(), view.getBottom());
            view.draw(canvas);

            ByteArrayOutputStream stream = new ByteArrayOutputStream();
            bitmap.compress(Bitmap.CompressFormat.PNG, COMPRESS_QUALITY, stream);

            return stream.toByteArray();
        } else {
            return new byte[0];
        }
    }

    /**
     * Gets color only if background is solid.
     */
    @NonNull
    public String getBackgroundColor(@NonNull View view) {
        Drawable background = view.getBackground();
        String backgroundColor = COLOR_TRANSPARENT;

        if (background instanceof ColorDrawable) {
            int color = ((ColorDrawable) background).getColor();
            backgroundColor = colorToHex(color);
        }

        return backgroundColor;
    }

    /**
     * Converts the supplied color into a hexadecimal color string, such as "2B23FF".
     */
    @NonNull
    public String colorToHex(int color) {
        return String.format("#%06X", COLOR_FORMAT & color);
    }

    /**
     * Detaches a view from it's parent, or no-op if already the case.
     *
     * @param view the view view we wanna verify is hanging.
     */
    public void verifyViewDetached(@NonNull View view) {
        ViewGroup parent = (ViewGroup) view.getParent();
        if (parent != null) {
            parent.removeView(view);
        }
    }
}