package com.contentsquare.android.internal.validator;

import android.content.ClipData;
import android.content.ClipboardManager;
import android.content.Context;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.UiThread;

import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.util.Strings;

/**
 * This logic inside this class verifies if the God Mode is enabled at runtime.
 * <p>
 * The actual verification is being made by checking if there is anything copied on the clipboard
 * for the device in use and comparing the clipboard value with the
 * {@link com.contentsquare.android.internal.config.ProjectConfiguration#getGodModeKey()}.
 */

public class GodModeClipboardValidator {
    private final Logger mLogger = new Logger(GodModeClipboardValidator.class.getSimpleName());
    @NonNull
    private final Context mContext;
    @Nullable
    private String mCurrentClipboardKey;

    /**
     * Creates a new GodModeClipboardValidator and initialize it.
     */
    @UiThread
    public GodModeClipboardValidator(@NonNull Context context) {
        mContext = context;
    }

    /**
     * Refreshes the currentClipboardKey by retrieving the latest one.
     */
    public void fetchClipboardData() {
        if (Looper.myLooper() == Looper.getMainLooper()) {
            mCurrentClipboardKey = null;
            ClipboardManager clipboard = (ClipboardManager) mContext
                    .getSystemService(Context.CLIPBOARD_SERVICE);
            if (clipboard == null) {
                return;
            }
            ClipData clipData = clipboard.getPrimaryClip();
            if (clipData == null) {
                return;
            }
            ClipData.Item item = clipData.getItemAt(0);
            if (item == null) {
                return;
            }
            CharSequence itemText = item.getText();
            if (itemText == null) {
                return;
            }
            mCurrentClipboardKey = itemText.toString();

        } else {
            mLogger.e("Failed to fetch Clip. Call was made outside of the main thread.");
            mCurrentClipboardKey = null;
        }
    }

    /**
     * Tells if the god mode key was provided and successfully verified.
     *
     * @return true if God Mode key was copied in the Clipboard
     */
    public boolean isGodModeKeyVerified(@Nullable String godModeKey) {
        // Protection against nullPointerExceptions and potential errors with empty keys
        if (Strings.isNullOrEmpty(godModeKey)) {
            return false;
        }

        boolean isGodModeValid = godModeKey.equals(mCurrentClipboardKey);

        mLogger.i("Godmode validation %s", isGodModeValid ? "passed." : "did not pass.");

        return isGodModeValid;
    }
}
