package com.contentsquare.android.internal.validator;

import static com.contentsquare.android.internal.preferences.PrefsHelper.LAST_SEGMENT;
import static com.contentsquare.android.internal.preferences.PrefsHelper.TRACKABLE;

import android.support.annotation.NonNull;
import android.support.annotation.VisibleForTesting;

import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.preferences.PrefsHelper;
import com.contentsquare.android.internal.util.RandomGenerator;


/**
 * An utility class which will compute the new sampling and decide if
 * this device is valid or not for tracking. The computed segment size will be stored and re - used
 * until the provided tracking sample rate is changing.
 */

public class InSampleIntervalValidator {

    /**
     * TAG to be used in the logger for this class.
     */
    public static final String TAG = "InSampleIntervalValidator";

    @NonNull
    private final PrefsHelper mPrefsHelper;
    @NonNull
    private final Logger mLogger;

    /**
     * Creates a new instance of this class.
     * The purpose of this created instance is to be kept as a singleton and provided
     * through Dagger injection.
     *
     * @param prefsHelper as the {@link PrefsHelper}
     * @param logger      as {@link Logger}
     */
    public InSampleIntervalValidator(@NonNull PrefsHelper prefsHelper,
                                     @NonNull Logger logger) {
        this.mPrefsHelper = prefsHelper;
        this.mLogger = logger;
    }

    /**
     * this method will calculate whether the api should run or not depending on
     * sample rate.
     * If a certain user is within the sample rate the run config will propose the api to run,
     * or not if otherwise.
     *
     * @param sampleRate as the sample rate based on which to apply/compute the rule
     * @return true if this device should be tracked.
     */
    public boolean isInSampleInterval(float sampleRate) {
        int segmentSize = computeSegmentSize(sampleRate);
        updateAudienceRestrictionsIfNeeded(mPrefsHelper, segmentSize);
        final boolean isInAudience = mPrefsHelper.getBoolean(TRACKABLE, true);
        mLogger.d("segment=%d, inAudience = %b", segmentSize, isInAudience);
        return isInAudience;
    }

    private void updateAudienceRestrictionsIfNeeded(PrefsHelper preferences, int segmentSize) {
        int lastSegment = preferences.getInt(LAST_SEGMENT, -1);
        if (lastSegment != segmentSize) {
            //segment size has changed which means we need to recalculate of this api should work.
            int sample = getRandomAllocation();
            preferences.putInt(LAST_SEGMENT, segmentSize);
            preferences.putBoolean(TRACKABLE, sample < segmentSize);
        }
    }

    private int computeSegmentSize(float sampleRate) {
        final int completeAudience = 100;
        return (int) (sampleRate * completeAudience);
    }

    @VisibleForTesting
    private int getRandomAllocation() {
        return RandomGenerator.getRandomAudiencePercentage();
    }
}
