/*
 * Decompiled with CFR 0.152.
 */
package name.wramner.httpclient;

import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.nio.charset.UnsupportedCharsetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;
import name.wramner.httpclient.EventRecorder;
import name.wramner.httpclient.HttpHeader;
import name.wramner.httpclient.HttpHeaderWithValue;
import name.wramner.httpclient.HttpHeaders;
import name.wramner.httpclient.HttpRequestBody;
import name.wramner.httpclient.HttpRequestMethod;
import name.wramner.httpclient.HttpResponse;

public class HttpClient {
    private static final int RECEIVE_BUFFER_SIZE = 1024;
    private static final String CRLF = "\r\n";
    private static final Charset HTTP_HEADER_CHARSET = Charset.forName("ISO-8859-1");
    public static final Charset HTTP_DEFAULT_CHARSET = Charset.forName("ISO-8859-1");
    private static final Set<HttpHeader> RESERVED_HEADERS = new HashSet<HttpHeader>(Arrays.asList(HttpHeaders.CONTENT_LENGTH, HttpHeaders.ACCEPT_ENCODING, HttpHeaders.CONNECTION, HttpHeaders.EXPECT, HttpHeaders.HOST));
    private final String _host;
    private final int _port;
    private final SSLSocketFactory _sslSocketFactory;
    private final int _connectTimeoutMillis;
    private final int _requestTimeoutMillis;
    private final boolean _use100Continue;

    HttpClient(String host, int port, SSLSocketFactory sslSocketFactory, int connectTimeoutMillis, int requestTimeoutMillis, boolean use100Continue) {
        this._host = host;
        this._port = port;
        this._sslSocketFactory = sslSocketFactory;
        this._connectTimeoutMillis = connectTimeoutMillis;
        this._requestTimeoutMillis = requestTimeoutMillis;
        this._use100Continue = use100Continue;
    }

    public HttpResponse sendRequest(HttpRequestMethod method, String url, HttpRequestBody body, HttpHeaderWithValue ... headers) throws IOException {
        return this.sendRequest(EventRecorder.NULL_RECORDER, method, url, body, headers);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public HttpResponse sendRequest(EventRecorder eventRecorder, HttpRequestMethod method, String url, HttpRequestBody body, HttpHeaderWithValue ... requestHeaders) throws IOException {
        eventRecorder.recordEvent(Event.ENTER_SEND_REQUEST);
        Socket socket = null;
        try {
            long deadlineMillis = System.currentTimeMillis() + (long)this._requestTimeoutMillis;
            socket = this.createSocket(eventRecorder);
            this.configureSocket(socket);
            StringBuilder sb = new StringBuilder();
            sb.append(method.name()).append(' ').append(url).append(" HTTP/1.1");
            sb.append(CRLF);
            byte[] requestBodyBytes = body.getBytes();
            this.appendRequestHeaders(sb, requestBodyBytes.length, requestHeaders);
            this.sendRequest(eventRecorder, socket, sb.toString().getBytes(HTTP_HEADER_CHARSET), requestBodyBytes);
            HttpResponse httpResponse = this.readResponse(eventRecorder, socket, deadlineMillis);
            return httpResponse;
        }
        finally {
            if (socket != null) {
                try {
                    socket.close();
                }
                catch (Exception e) {}
            }
            eventRecorder.recordEvent(Event.EXIT_SEND_REQUEST);
        }
    }

    private HttpResponse readResponse(EventRecorder eventRecorder, Socket socket, long deadlineMillis) throws IOException {
        List<HttpHeaderWithValue> responseHeaders;
        Integer contentLength;
        eventRecorder.recordEvent(Event.READING_RESPONSE);
        byte[] buffer = new byte[1024];
        InputStream in = socket.getInputStream();
        int totalRead = 0;
        int bodyPosition = 0;
        while (totalRead < buffer.length && bodyPosition == 0) {
            this.updateSocketTimeout(socket, deadlineMillis);
            int read = in.read(buffer, totalRead, buffer.length - totalRead);
            if (read == -1) {
                throw new EOFException("Unexpected end of response after " + totalRead + " bytes");
            }
            if (read <= 0) continue;
            bodyPosition = HttpClient.findBodyPosition(buffer, totalRead += read);
        }
        if (bodyPosition == 0) {
            throw new IOException("More than " + buffer.length + " bytes read before body!");
        }
        int endOfStatusLine = HttpClient.findEndOfLine(buffer, 0, bodyPosition);
        if (endOfStatusLine == 0) {
            throw new IllegalStateException("Found CRLFCRLF but not CRLF!?!");
        }
        int httpResponseCode = this.parseHttpStatusCode(buffer, endOfStatusLine);
        ByteArrayOutputStream bodyOutputStream = new ByteArrayOutputStream();
        if (bodyPosition < totalRead) {
            bodyOutputStream.write(buffer, bodyPosition, totalRead - bodyPosition);
        }
        if ((contentLength = this.findContentLength(responseHeaders = this.parseHeaders(buffer, endOfStatusLine + 2, bodyPosition))) != null) {
            int read;
            for (int remainingContentLength = contentLength - (totalRead - bodyPosition); remainingContentLength > 0; remainingContentLength -= read) {
                this.updateSocketTimeout(socket, deadlineMillis);
                read = in.read(buffer, 0, Math.min(buffer.length, remainingContentLength));
                if (read == -1) {
                    throw new EOFException("Partial response, " + remainingContentLength + " bytes missing");
                }
                if (read == 0) {
                    throw new SocketTimeoutException("Timeout reading response body");
                }
                bodyOutputStream.write(buffer, 0, read);
            }
        } else {
            while (true) {
                this.updateSocketTimeout(socket, deadlineMillis);
                int read = in.read(buffer, 0, buffer.length);
                if (read == -1) break;
                if (read == 0) {
                    throw new SocketTimeoutException("Timeout reading response body");
                }
                bodyOutputStream.write(buffer, 0, read);
            }
        }
        eventRecorder.recordEvent(Event.READ_RESPONSE);
        return new HttpResponse(httpResponseCode, responseHeaders, bodyOutputStream.toByteArray());
    }

    private Integer findContentLength(List<HttpHeaderWithValue> responseHeaders) {
        for (HttpHeaderWithValue headerWithValue : responseHeaders) {
            if (!HttpHeaders.CONTENT_LENGTH.equals(headerWithValue.getHeader())) continue;
            return Integer.valueOf(headerWithValue.getValue());
        }
        return null;
    }

    private void appendRequestHeaders(StringBuilder sb, int contentLength, HttpHeaderWithValue ... requestHeaders) {
        ArrayList<HttpHeaderWithValue> requestHeaderList = new ArrayList<HttpHeaderWithValue>();
        for (HttpHeaderWithValue headerWithValue : requestHeaders) {
            if (RESERVED_HEADERS.contains(headerWithValue.getHeader())) continue;
            requestHeaderList.add(headerWithValue);
        }
        requestHeaderList.add(HttpHeaders.CONTENT_LENGTH.withValue(String.valueOf(contentLength)));
        requestHeaderList.add(HttpHeaders.ACCEPT_ENCODING.withValue("identity"));
        requestHeaderList.add(HttpHeaders.CONNECTION.withValue("close"));
        requestHeaderList.add(HttpHeaders.HOST.withValue(this._host + ":" + this._port));
        if (this._use100Continue) {
            requestHeaderList.add(HttpHeaders.EXPECT.withValue("100-continue"));
        }
        for (HttpHeaderWithValue headerWithValue : requestHeaderList) {
            sb.append(headerWithValue.getHeader().getName()).append(": ");
            sb.append(headerWithValue.getValue());
            sb.append(CRLF);
        }
        sb.append(CRLF);
    }

    private int parseHttpStatusCode(byte[] buffer, int endOfStatusLine) throws IOException {
        String statusLine = new String(buffer, 0, endOfStatusLine, HTTP_HEADER_CHARSET);
        String[] statusFields = statusLine.split(" ");
        if (statusFields.length >= 3 && statusFields[0].startsWith("HTTP/")) {
            return Integer.parseInt(statusFields[1]);
        }
        throw new IOException("Invalid HTTP response status line: " + statusLine);
    }

    private void sendRequest(EventRecorder eventRecorder, Socket socket, byte[] requestHeader, byte[] requestBody) throws IOException {
        eventRecorder.recordEvent(Event.SENDING_REQUEST);
        OutputStream out = socket.getOutputStream();
        out.write(requestHeader);
        if (this._use100Continue) {
            out.flush();
            eventRecorder.recordEvent(Event.SENT_HEADERS_WAITING_FOR_100_CONTINUE);
            Thread.yield();
            this.waitFor100Continue(socket);
            eventRecorder.recordEvent(Event.RECEIVED_100_CONTINUE);
        }
        out.write(requestBody);
        out.flush();
        eventRecorder.recordEvent(Event.SENT_REQUEST);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void waitFor100Continue(Socket socket) throws IOException {
        InputStream in = socket.getInputStream();
        ByteArrayOutputStream byteStream = new ByteArrayOutputStream();
        boolean foundCr = false;
        boolean found100Continue = false;
        int c = in.read();
        while (true) {
            if (c == -1) {
                throw new EOFException("End of file waiting for 100-continue!");
            }
            if (c == 13) {
                foundCr = true;
            } else if (c == 10) {
                if (foundCr) {
                    foundCr = false;
                    byte[] lineAsBytes = byteStream.toByteArray();
                    if (lineAsBytes.length == 0) {
                        if (found100Continue) return;
                        throw new IOException("Unexpected blank line before 100 continue!");
                    }
                    String line = new String(lineAsBytes, HTTP_HEADER_CHARSET);
                    if (!line.startsWith("HTTP/1.1 100")) throw new IOException("Unexpected response waiting for 100 continue: " + line);
                    byteStream = new ByteArrayOutputStream();
                    found100Continue = true;
                }
            } else {
                foundCr = false;
                byteStream.write(c);
            }
            c = in.read();
        }
    }

    private void configureSocket(Socket socket) throws SocketException {
        socket.setTcpNoDelay(true);
        socket.setKeepAlive(false);
        socket.setSoTimeout(this._requestTimeoutMillis);
    }

    private void updateSocketTimeout(Socket socket, long deadlineMillis) throws SocketTimeoutException, SocketException {
        long remainingTimeMillis = deadlineMillis - System.currentTimeMillis();
        if (remainingTimeMillis <= 0L) {
            throw new SocketTimeoutException("Request timed out");
        }
        socket.setSoTimeout((int)remainingTimeMillis);
    }

    private List<HttpHeaderWithValue> parseHeaders(byte[] buffer, int startPos, int endPos) {
        ArrayList<HttpHeaderWithValue> responseHeaders = new ArrayList<HttpHeaderWithValue>();
        int lineStartPos = startPos;
        for (int lineEndPos = 1; lineEndPos < endPos; ++lineEndPos) {
            if (buffer[lineEndPos] != 10 || buffer[lineEndPos - 1] != 13) continue;
            if (lineEndPos > lineStartPos + 1) {
                String headerLine = new String(buffer, lineStartPos, lineEndPos - 1 - lineStartPos, HTTP_HEADER_CHARSET);
                int separatorPos = headerLine.indexOf(":");
                if (separatorPos == -1) {
                    responseHeaders.add(new HttpHeader(headerLine).withValue(""));
                } else if (separatorPos > 0) {
                    String headerKey = headerLine.substring(0, separatorPos).toLowerCase();
                    String headerValue = (separatorPos < headerLine.length() ? headerLine.substring(separatorPos + 1) : "").trim();
                    responseHeaders.add(new HttpHeader(headerKey).withValue(headerValue));
                }
            }
            lineStartPos = lineEndPos + 1;
        }
        return responseHeaders;
    }

    static int findEndOfLine(byte[] buffer, int startPos, int endPos) {
        for (int pos = startPos; pos < endPos - 1; ++pos) {
            if (buffer[pos] != 13 || buffer[pos + 1] != 10) continue;
            return pos;
        }
        return -1;
    }

    static int findBodyPosition(byte[] buffer, int endPos) {
        int bodyPos = 0;
        while (bodyPos + 3 < endPos) {
            if (buffer[bodyPos] == 13 && buffer[bodyPos + 1] == 10 && buffer[bodyPos + 2] == 13 && buffer[bodyPos + 3] == 10) {
                return bodyPos + 4;
            }
            ++bodyPos;
        }
        return 0;
    }

    public static Charset extractCharsetFromContentType(String contentType) {
        int startIndex;
        int charsetIndex = contentType.indexOf("charset");
        if (charsetIndex != -1 && (startIndex = contentType.indexOf("=", charsetIndex) + 1) > charsetIndex) {
            int nextCommaIndex = contentType.indexOf(",", startIndex);
            int nextSemiColonIndex = contentType.indexOf(";", startIndex);
            int endIndex = nextCommaIndex > startIndex && nextSemiColonIndex > startIndex ? Math.min(nextCommaIndex, nextSemiColonIndex) : (nextCommaIndex > startIndex ? nextCommaIndex : (nextSemiColonIndex > startIndex ? nextSemiColonIndex : contentType.length()));
            String charsetName = contentType.substring(startIndex, endIndex);
            try {
                return Charset.forName(charsetName);
            }
            catch (UnsupportedCharsetException e) {
                return HTTP_DEFAULT_CHARSET;
            }
        }
        return HTTP_DEFAULT_CHARSET;
    }

    private Socket createSocket(EventRecorder recorder) throws UnknownHostException, IOException {
        Socket nonSslSocket = new Socket();
        recorder.recordEvent(Event.CONNECTING);
        nonSslSocket.connect(new InetSocketAddress(this._host, this._port), this._connectTimeoutMillis);
        recorder.recordEvent(Event.CONNECTED);
        if (this._sslSocketFactory != null) {
            SSLSocket socket = (SSLSocket)this._sslSocketFactory.createSocket(nonSslSocket, this._host, this._port, true);
            socket.setUseClientMode(true);
            socket.startHandshake();
            recorder.recordEvent(Event.SSL_HANDSHAKE_COMPLETE);
            return socket;
        }
        return nonSslSocket;
    }

    public static enum Event {
        ENTER_SEND_REQUEST,
        CONNECTING,
        CONNECTED,
        SSL_HANDSHAKE_COMPLETE,
        SENDING_REQUEST,
        SENT_HEADERS_WAITING_FOR_100_CONTINUE,
        RECEIVED_100_CONTINUE,
        SENT_REQUEST,
        READING_RESPONSE,
        READ_RESPONSE,
        EXIT_SEND_REQUEST;

    }
}

