/*
 * Decompiled with CFR 0.152.
 */
package alluxio.hadoop;

import alluxio.AlluxioURI;
import alluxio.Configuration;
import alluxio.client.ClientContext;
import alluxio.client.block.BlockStoreContext;
import alluxio.client.file.FileOutStream;
import alluxio.client.file.FileSystem;
import alluxio.client.file.FileSystemContext;
import alluxio.client.file.FileSystemMasterClient;
import alluxio.client.file.URIStatus;
import alluxio.client.file.options.CreateDirectoryOptions;
import alluxio.client.file.options.CreateFileOptions;
import alluxio.client.file.options.DeleteOptions;
import alluxio.client.file.options.SetAttributeOptions;
import alluxio.client.lineage.LineageContext;
import alluxio.exception.AlluxioException;
import alluxio.exception.ConnectionFailedException;
import alluxio.exception.ExceptionMessage;
import alluxio.exception.FileDoesNotExistException;
import alluxio.exception.InvalidPathException;
import alluxio.hadoop.ConfUtils;
import alluxio.hadoop.HadoopUtils;
import alluxio.hadoop.HdfsFileInputStream;
import alluxio.util.CommonUtils;
import alluxio.wire.BlockLocation;
import alluxio.wire.FileBlockInfo;
import alluxio.wire.WorkerNetAddress;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import javax.annotation.concurrent.NotThreadSafe;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.util.Progressable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
abstract class AbstractFileSystem
extends FileSystem {
    public static final String FIRST_COM_PATH = "alluxio_dep/";
    public static final String RECOMPUTE_PATH = "alluxio_recompute/";
    private static final Logger LOG = LoggerFactory.getLogger((String)"alluxio.logger.type");
    private static final int BLOCK_REPLICATION_CONSTANT = 3;
    private String mUnderFSAddress;
    private URI mUri = null;
    private Path mWorkingDir = new Path("/");
    private FileSystem.Statistics mStatistics = null;
    private alluxio.client.file.FileSystem mFileSystem = null;
    private String mAlluxioHeader = null;

    AbstractFileSystem() {
    }

    public FSDataOutputStream append(Path cPath, int bufferSize, Progressable progress) throws IOException {
        LOG.info("append({}, {}, {})", new Object[]{cPath, bufferSize, progress});
        if (this.mStatistics != null) {
            this.mStatistics.incrementWriteOps(1);
        }
        AlluxioURI path = new AlluxioURI(HadoopUtils.getPathWithoutScheme(cPath));
        try {
            if (!this.mFileSystem.exists(path)) {
                return new FSDataOutputStream((OutputStream)this.mFileSystem.createFile(path), this.mStatistics);
            }
            throw new IOException(ExceptionMessage.FILE_ALREADY_EXISTS.getMessage(new Object[]{path}));
        }
        catch (InvalidPathException e) {
            throw new IOException(e);
        }
        catch (AlluxioException e) {
            throw new IOException(e);
        }
    }

    public void close() throws IOException {
        super.close();
    }

    public FSDataOutputStream create(Path cPath, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        LOG.info("create({}, {}, {}, {}, {}, {}, {})", new Object[]{cPath, permission, overwrite, bufferSize, replication, blockSize, progress});
        if (this.mStatistics != null) {
            this.mStatistics.incrementWriteOps(1);
        }
        AlluxioURI path = new AlluxioURI(HadoopUtils.getPathWithoutScheme(cPath));
        try {
            if (this.mFileSystem.exists(path)) {
                if (!overwrite) {
                    throw new IOException(ExceptionMessage.FILE_ALREADY_EXISTS.getMessage(new Object[]{cPath.toString()}));
                }
                if (this.mFileSystem.getStatus(path).isFolder()) {
                    throw new IOException(ExceptionMessage.FILE_CREATE_IS_DIRECTORY.getMessage(new Object[]{cPath.toString()}));
                }
                this.mFileSystem.delete(path);
            }
        }
        catch (AlluxioException e) {
            throw new IOException(e);
        }
        CreateFileOptions options = CreateFileOptions.defaults().setBlockSizeBytes(blockSize);
        try {
            FileOutStream outStream = this.mFileSystem.createFile(path, options);
            return new FSDataOutputStream((OutputStream)outStream, this.mStatistics);
        }
        catch (AlluxioException e) {
            throw new IOException(e);
        }
    }

    @Deprecated
    public FSDataOutputStream createNonRecursive(Path cPath, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        AlluxioURI parentPath = new AlluxioURI(HadoopUtils.getPathWithoutScheme(cPath.getParent()));
        this.ensureExists(parentPath);
        return this.create(cPath, permission, overwrite, bufferSize, replication, blockSize, progress);
    }

    @Deprecated
    public boolean delete(Path path) throws IOException {
        return this.delete(path, true);
    }

    public boolean delete(Path cPath, boolean recursive) throws IOException {
        LOG.info("delete({}, {})", (Object)cPath, (Object)recursive);
        if (this.mStatistics != null) {
            this.mStatistics.incrementWriteOps(1);
        }
        AlluxioURI path = new AlluxioURI(HadoopUtils.getPathWithoutScheme(cPath));
        DeleteOptions options = DeleteOptions.defaults().setRecursive(recursive);
        try {
            this.mFileSystem.delete(path, options);
            return true;
        }
        catch (InvalidPathException e) {
            LOG.info("delete failed: {}", (Object)e.getMessage());
            return false;
        }
        catch (FileDoesNotExistException e) {
            LOG.info("delete failed: {}", (Object)e.getMessage());
            return false;
        }
        catch (AlluxioException e) {
            throw new IOException(e);
        }
    }

    public long getDefaultBlockSize() {
        return ClientContext.getConf().getBytes("alluxio.user.block.size.bytes.default");
    }

    public org.apache.hadoop.fs.BlockLocation[] getFileBlockLocations(FileStatus file, long start, long len) throws IOException {
        if (file == null) {
            return null;
        }
        if (this.mStatistics != null) {
            this.mStatistics.incrementReadOps(1);
        }
        AlluxioURI path = new AlluxioURI(HadoopUtils.getPathWithoutScheme(file.getPath()));
        try {
            URIStatus status = this.mFileSystem.getStatus(path);
        }
        catch (AlluxioException e) {
            throw new IOException(e);
        }
        List<FileBlockInfo> blocks = this.getFileBlocks(path);
        ArrayList<org.apache.hadoop.fs.BlockLocation> blockLocations = new ArrayList<org.apache.hadoop.fs.BlockLocation>();
        for (int k = 0; k < blocks.size(); ++k) {
            FileBlockInfo info = blocks.get(k);
            long offset = info.getOffset();
            long end = offset + info.getBlockInfo().getLength();
            if (end < start || offset > start + len) continue;
            ArrayList<String> names = new ArrayList<String>();
            ArrayList<String> hosts = new ArrayList<String>();
            ArrayList addrs = Lists.newArrayList();
            for (BlockLocation location : info.getBlockInfo().getLocations()) {
                addrs.add(location.getWorkerAddress());
            }
            addrs.addAll(info.getUfsLocations());
            for (WorkerNetAddress addr : addrs) {
                String name = addr.getHost() + ":" + addr.getDataPort();
                LOG.debug("getFileBlockLocations : adding name : {}", (Object)name);
                names.add(name);
                hosts.add(addr.getHost());
            }
            blockLocations.add(new org.apache.hadoop.fs.BlockLocation(CommonUtils.toStringArray(names), CommonUtils.toStringArray(hosts), offset, info.getBlockInfo().getLength()));
        }
        org.apache.hadoop.fs.BlockLocation[] ret = new org.apache.hadoop.fs.BlockLocation[blockLocations.size()];
        blockLocations.toArray(ret);
        return ret;
    }

    public FileStatus getFileStatus(Path path) throws IOException {
        URIStatus fileStatus;
        AlluxioURI tPath = new AlluxioURI(HadoopUtils.getPathWithoutScheme(path));
        Path hdfsPath = HadoopUtils.getHDFSPath(tPath, this.mUnderFSAddress);
        LOG.info("getFileStatus({}): HDFS Path: {} Alluxio Path: {}{}", new Object[]{path, hdfsPath, this.mAlluxioHeader, tPath});
        if (this.mStatistics != null) {
            this.mStatistics.incrementReadOps(1);
        }
        try {
            fileStatus = this.mFileSystem.getStatus(tPath);
        }
        catch (FileDoesNotExistException e) {
            throw new FileNotFoundException(e.getMessage());
        }
        catch (AlluxioException e) {
            throw new IOException(e);
        }
        FileStatus ret = new FileStatus(fileStatus.getLength(), fileStatus.isFolder(), 3, fileStatus.getBlockSizeBytes(), fileStatus.getCreationTimeMs(), fileStatus.getCreationTimeMs(), new FsPermission((short)fileStatus.getPermission()), fileStatus.getUserName(), fileStatus.getGroupName(), new Path(this.mAlluxioHeader + tPath));
        return ret;
    }

    public void setOwner(Path path, String username, String groupname) throws IOException {
        AlluxioURI tPath = new AlluxioURI(HadoopUtils.getPathWithoutScheme(path));
        Path hdfsPath = HadoopUtils.getHDFSPath(tPath, this.mUnderFSAddress);
        LOG.info("setOwner({},{},{}) HDFS Path: {} Alluxio Path: {}{}", new Object[]{path, username, groupname, hdfsPath, this.mAlluxioHeader, tPath});
        try {
            SetAttributeOptions options = SetAttributeOptions.defaults();
            boolean ownerOrGroupChanged = false;
            if (username != null && !username.isEmpty()) {
                options.setOwner(username).setRecursive(false);
                ownerOrGroupChanged = true;
            }
            if (groupname != null && !groupname.isEmpty()) {
                options.setGroup(groupname).setRecursive(false);
                ownerOrGroupChanged = true;
            }
            if (ownerOrGroupChanged) {
                this.mFileSystem.setAttribute(tPath, options);
            }
        }
        catch (AlluxioException e) {
            throw new IOException(e);
        }
    }

    public void setPermission(Path path, FsPermission permission) throws IOException {
        AlluxioURI tPath = new AlluxioURI(HadoopUtils.getPathWithoutScheme(path));
        Path hdfsPath = HadoopUtils.getHDFSPath(tPath, this.mUnderFSAddress);
        LOG.info("setPermission({},{}) HDFS Path: {} Alluxio Path: {}{}", new Object[]{path, permission.toString(), hdfsPath, this.mAlluxioHeader, tPath});
        try {
            SetAttributeOptions options = SetAttributeOptions.defaults().setPermission(permission.toShort()).setRecursive(false);
            this.mFileSystem.setAttribute(tPath, options);
        }
        catch (AlluxioException e) {
            throw new IOException(e);
        }
    }

    public abstract String getScheme();

    public URI getUri() {
        return this.mUri;
    }

    public Path getWorkingDirectory() {
        LOG.info("getWorkingDirectory: {}", (Object)this.mWorkingDir);
        return this.mWorkingDir;
    }

    public void initialize(URI uri, org.apache.hadoop.conf.Configuration conf) throws IOException {
        Preconditions.checkNotNull((Object)uri.getHost(), (Object)"URI hostname must not be null");
        Preconditions.checkNotNull((Object)uri.getPort(), (Object)"URI port must not be null");
        super.initialize(uri, conf);
        LOG.info("initialize({}, {}). Connecting to Alluxio: {}", new Object[]{uri, conf, uri.toString()});
        HadoopUtils.addS3Credentials(conf);
        this.setConf(conf);
        this.mAlluxioHeader = this.getScheme() + "://" + uri.getHost() + ":" + uri.getPort();
        this.mStatistics = this.statistics;
        Configuration siteConf = ConfUtils.loadFromHadoopConfiguration(conf);
        if (siteConf != null) {
            ClientContext.getConf().merge(siteConf);
        }
        ClientContext.getConf().set("alluxio.master.hostname", uri.getHost());
        ClientContext.getConf().set("alluxio.master.port", Integer.toString(uri.getPort()));
        ClientContext.getConf().set("alluxio.zookeeper.enabled", Boolean.toString(this.isZookeeperMode()));
        ClientContext.init();
        FileSystemContext.INSTANCE.reset();
        BlockStoreContext.INSTANCE.reset();
        LineageContext.INSTANCE.reset();
        this.mFileSystem = FileSystem.Factory.get();
        this.mUri = URI.create(this.mAlluxioHeader);
        this.mUnderFSAddress = this.getUfsAddress();
        LOG.info("{} {} {}", new Object[]{this.mAlluxioHeader, this.mUri, this.mUnderFSAddress});
    }

    protected abstract boolean isZookeeperMode();

    public FileStatus[] listStatus(Path path) throws IOException {
        List<URIStatus> statuses;
        AlluxioURI tPath = new AlluxioURI(HadoopUtils.getPathWithoutScheme(path));
        Path hdfsPath = HadoopUtils.getHDFSPath(tPath, this.mUnderFSAddress);
        LOG.info("listStatus({}): HDFS Path: {}", (Object)path, (Object)hdfsPath);
        if (this.mStatistics != null) {
            this.mStatistics.incrementReadOps(1);
        }
        try {
            statuses = this.mFileSystem.listStatus(tPath);
        }
        catch (AlluxioException e) {
            throw new IOException(e);
        }
        FileStatus[] ret = new FileStatus[statuses.size()];
        for (int k = 0; k < statuses.size(); ++k) {
            URIStatus status = statuses.get(k);
            ret[k] = new FileStatus(status.getLength(), status.isFolder(), 3, status.getBlockSizeBytes(), status.getCreationTimeMs(), status.getCreationTimeMs(), null, null, null, new Path(this.mAlluxioHeader + status.getPath()));
        }
        return ret;
    }

    public boolean mkdirs(Path cPath, FsPermission permission) throws IOException {
        LOG.info("mkdirs({}, {})", (Object)cPath, (Object)permission);
        if (this.mStatistics != null) {
            this.mStatistics.incrementWriteOps(1);
        }
        AlluxioURI path = new AlluxioURI(HadoopUtils.getPathWithoutScheme(cPath));
        CreateDirectoryOptions options = CreateDirectoryOptions.defaults().setRecursive(true).setAllowExists(true);
        try {
            this.mFileSystem.createDirectory(path, options);
            return true;
        }
        catch (AlluxioException e) {
            throw new IOException(e);
        }
    }

    public FSDataInputStream open(Path cPath, int bufferSize) throws IOException {
        LOG.info("open({}, {})", (Object)cPath, (Object)bufferSize);
        if (this.mStatistics != null) {
            this.mStatistics.incrementReadOps(1);
        }
        AlluxioURI path = new AlluxioURI(HadoopUtils.getPathWithoutScheme(cPath));
        return new FSDataInputStream((InputStream)new HdfsFileInputStream(path, HadoopUtils.getHDFSPath(path, this.mUnderFSAddress), this.getConf(), bufferSize, this.mStatistics));
    }

    public boolean rename(Path src, Path dst) throws IOException {
        URIStatus dstStatus;
        LOG.info("rename({}, {})", (Object)src, (Object)dst);
        if (this.mStatistics != null) {
            this.mStatistics.incrementWriteOps(1);
        }
        AlluxioURI srcPath = new AlluxioURI(HadoopUtils.getPathWithoutScheme(src));
        AlluxioURI dstPath = new AlluxioURI(HadoopUtils.getPathWithoutScheme(dst));
        this.ensureExists(srcPath);
        try {
            dstStatus = this.mFileSystem.getStatus(dstPath);
        }
        catch (IOException e) {
            dstStatus = null;
        }
        catch (AlluxioException e) {
            dstStatus = null;
        }
        if (dstStatus != null && dstStatus.isFolder()) {
            dstPath = dstPath.join(srcPath.getName());
        }
        try {
            this.mFileSystem.rename(srcPath, dstPath);
            return true;
        }
        catch (IOException e) {
            LOG.error("Failed to rename {} to {}", new Object[]{src, dst, e});
            return false;
        }
        catch (AlluxioException e) {
            LOG.error("Failed to rename {} to {}", new Object[]{src, dst, e});
            return false;
        }
    }

    public void setWorkingDirectory(Path path) {
        LOG.info("setWorkingDirectory({})", (Object)path);
        this.mWorkingDir = path.isAbsolute() ? path : new Path(this.mWorkingDir, path);
    }

    private void ensureExists(AlluxioURI path) throws IOException {
        try {
            this.mFileSystem.getStatus(path);
        }
        catch (AlluxioException e) {
            throw new IOException(e);
        }
    }

    private List<FileBlockInfo> getFileBlocks(AlluxioURI path) throws IOException {
        FileSystemMasterClient master = FileSystemContext.INSTANCE.acquireMasterClient();
        try {
            List<FileBlockInfo> list = master.getFileBlockInfoList(path);
            return list;
        }
        catch (AlluxioException e) {
            throw new IOException(e);
        }
        finally {
            FileSystemContext.INSTANCE.releaseMasterClient(master);
        }
    }

    private String getUfsAddress() throws IOException {
        FileSystemMasterClient master = FileSystemContext.INSTANCE.acquireMasterClient();
        try {
            String string = master.getUfsAddress();
            return string;
        }
        catch (ConnectionFailedException e) {
            throw new IOException(e);
        }
        finally {
            FileSystemContext.INSTANCE.releaseMasterClient(master);
        }
    }
}

