/*
 * Decompiled with CFR 0.152.
 */
package alluxio.worker.block.meta;

import alluxio.exception.BlockAlreadyExistsException;
import alluxio.exception.BlockDoesNotExistException;
import alluxio.exception.ExceptionMessage;
import alluxio.exception.InvalidWorkerStateException;
import alluxio.exception.WorkerOutOfSpaceException;
import alluxio.worker.block.BlockStoreLocation;
import alluxio.worker.block.meta.BlockMeta;
import alluxio.worker.block.meta.StorageTier;
import alluxio.worker.block.meta.TempBlockMeta;
import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.concurrent.NotThreadSafe;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
public final class StorageDir {
    private static final Logger LOG = LoggerFactory.getLogger((String)"alluxio.logger.type");
    private final long mCapacityBytes;
    private Map<Long, BlockMeta> mBlockIdToBlockMap;
    private Map<Long, TempBlockMeta> mBlockIdToTempBlockMap;
    private Map<Long, Set<Long>> mSessionIdToTempBlockIdsMap;
    private AtomicLong mAvailableBytes;
    private AtomicLong mCommittedBytes;
    private String mDirPath;
    private int mDirIndex;
    private StorageTier mTier;

    private StorageDir(StorageTier tier, int dirIndex, long capacityBytes, String dirPath) {
        this.mTier = (StorageTier)Preconditions.checkNotNull((Object)tier);
        this.mDirIndex = dirIndex;
        this.mCapacityBytes = capacityBytes;
        this.mAvailableBytes = new AtomicLong(capacityBytes);
        this.mCommittedBytes = new AtomicLong(0L);
        this.mDirPath = dirPath;
        this.mBlockIdToBlockMap = new HashMap<Long, BlockMeta>(200);
        this.mBlockIdToTempBlockMap = new HashMap<Long, TempBlockMeta>(200);
        this.mSessionIdToTempBlockIdsMap = new HashMap<Long, Set<Long>>(200);
    }

    public static StorageDir newStorageDir(StorageTier tier, int dirIndex, long capacityBytes, String dirPath) throws BlockAlreadyExistsException, IOException, WorkerOutOfSpaceException {
        StorageDir dir = new StorageDir(tier, dirIndex, capacityBytes, dirPath);
        dir.initializeMeta();
        return dir;
    }

    private void initializeMeta() throws BlockAlreadyExistsException, IOException, WorkerOutOfSpaceException {
        alluxio.util.io.FileUtils.createStorageDirPath((String)this.mDirPath);
        File dir = new File(this.mDirPath);
        File[] paths = dir.listFiles();
        if (paths == null) {
            return;
        }
        for (File path : paths) {
            if (!path.isFile()) {
                LOG.error("{} in StorageDir is not a file", (Object)path.getAbsolutePath());
                try {
                    FileUtils.deleteDirectory((File)path);
                }
                catch (IOException e) {
                    LOG.error("can not delete directory {}", (Object)path.getAbsolutePath(), (Object)e);
                }
                continue;
            }
            try {
                long blockId = Long.parseLong(path.getName());
                this.addBlockMeta(new BlockMeta(blockId, path.length(), this));
            }
            catch (NumberFormatException e) {
                LOG.error("filename of {} in StorageDir can not be parsed into long", (Object)path.getAbsolutePath(), (Object)e);
                if (path.delete()) {
                    LOG.warn("file {} has been deleted", (Object)path.getAbsolutePath());
                    continue;
                }
                LOG.error("can not delete file {}", (Object)path.getAbsolutePath());
            }
        }
    }

    public long getCapacityBytes() {
        return this.mCapacityBytes;
    }

    public long getAvailableBytes() {
        return this.mAvailableBytes.get();
    }

    public long getCommittedBytes() {
        return this.mCommittedBytes.get();
    }

    public String getDirPath() {
        return this.mDirPath;
    }

    public StorageTier getParentTier() {
        return this.mTier;
    }

    public int getDirIndex() {
        return this.mDirIndex;
    }

    public List<Long> getBlockIds() {
        return new ArrayList<Long>(this.mBlockIdToBlockMap.keySet());
    }

    public List<BlockMeta> getBlocks() {
        return new ArrayList<BlockMeta>(this.mBlockIdToBlockMap.values());
    }

    public boolean hasBlockMeta(long blockId) {
        return this.mBlockIdToBlockMap.containsKey(blockId);
    }

    public boolean hasTempBlockMeta(long blockId) {
        return this.mBlockIdToTempBlockMap.containsKey(blockId);
    }

    public BlockMeta getBlockMeta(long blockId) throws BlockDoesNotExistException {
        BlockMeta blockMeta = this.mBlockIdToBlockMap.get(blockId);
        if (blockMeta == null) {
            throw new BlockDoesNotExistException(ExceptionMessage.BLOCK_META_NOT_FOUND, new Object[]{blockId});
        }
        return blockMeta;
    }

    public TempBlockMeta getTempBlockMeta(long blockId) throws BlockDoesNotExistException {
        TempBlockMeta tempBlockMeta = this.mBlockIdToTempBlockMap.get(blockId);
        if (tempBlockMeta == null) {
            throw new BlockDoesNotExistException(ExceptionMessage.TEMP_BLOCK_META_NOT_FOUND, new Object[]{blockId});
        }
        return tempBlockMeta;
    }

    public void addBlockMeta(BlockMeta blockMeta) throws WorkerOutOfSpaceException, BlockAlreadyExistsException {
        Preconditions.checkNotNull((Object)blockMeta);
        long blockId = blockMeta.getBlockId();
        long blockSize = blockMeta.getBlockSize();
        if (this.getAvailableBytes() < blockSize) {
            throw new WorkerOutOfSpaceException(ExceptionMessage.NO_SPACE_FOR_BLOCK_META, new Object[]{blockId, blockSize, this.getAvailableBytes(), blockMeta.getBlockLocation().tierAlias()});
        }
        if (this.hasBlockMeta(blockId)) {
            throw new BlockAlreadyExistsException(ExceptionMessage.ADD_EXISTING_BLOCK, new Object[]{blockId, blockMeta.getBlockLocation().tierAlias()});
        }
        this.mBlockIdToBlockMap.put(blockId, blockMeta);
        this.reserveSpace(blockSize, true);
    }

    public void addTempBlockMeta(TempBlockMeta tempBlockMeta) throws WorkerOutOfSpaceException, BlockAlreadyExistsException {
        Preconditions.checkNotNull((Object)tempBlockMeta);
        long sessionId = tempBlockMeta.getSessionId();
        long blockId = tempBlockMeta.getBlockId();
        long blockSize = tempBlockMeta.getBlockSize();
        if (this.getAvailableBytes() < blockSize) {
            throw new WorkerOutOfSpaceException(ExceptionMessage.NO_SPACE_FOR_BLOCK_META, new Object[]{blockId, blockSize, this.getAvailableBytes(), tempBlockMeta.getBlockLocation().tierAlias()});
        }
        if (this.hasTempBlockMeta(blockId)) {
            throw new BlockAlreadyExistsException(ExceptionMessage.ADD_EXISTING_BLOCK, new Object[]{blockId, tempBlockMeta.getBlockLocation().tierAlias()});
        }
        this.mBlockIdToTempBlockMap.put(blockId, tempBlockMeta);
        Set<Long> sessionTempBlocks = this.mSessionIdToTempBlockIdsMap.get(sessionId);
        if (sessionTempBlocks == null) {
            this.mSessionIdToTempBlockIdsMap.put(sessionId, Sets.newHashSet((Object[])new Long[]{blockId}));
        } else {
            sessionTempBlocks.add(blockId);
        }
        this.reserveSpace(blockSize, false);
    }

    public void removeBlockMeta(BlockMeta blockMeta) throws BlockDoesNotExistException {
        Preconditions.checkNotNull((Object)blockMeta);
        long blockId = blockMeta.getBlockId();
        BlockMeta deletedBlockMeta = this.mBlockIdToBlockMap.remove(blockId);
        if (deletedBlockMeta == null) {
            throw new BlockDoesNotExistException(ExceptionMessage.BLOCK_META_NOT_FOUND, new Object[]{blockId});
        }
        this.reclaimSpace(blockMeta.getBlockSize(), true);
    }

    public void removeTempBlockMeta(TempBlockMeta tempBlockMeta) throws BlockDoesNotExistException {
        Preconditions.checkNotNull((Object)tempBlockMeta);
        long blockId = tempBlockMeta.getBlockId();
        long sessionId = tempBlockMeta.getSessionId();
        TempBlockMeta deletedTempBlockMeta = this.mBlockIdToTempBlockMap.remove(blockId);
        if (deletedTempBlockMeta == null) {
            throw new BlockDoesNotExistException(ExceptionMessage.BLOCK_META_NOT_FOUND, new Object[]{blockId});
        }
        Set<Long> sessionBlocks = this.mSessionIdToTempBlockIdsMap.get(sessionId);
        if (sessionBlocks == null || !sessionBlocks.contains(blockId)) {
            throw new BlockDoesNotExistException(ExceptionMessage.BLOCK_NOT_FOUND_FOR_SESSION, new Object[]{blockId, this.mTier.getTierAlias(), sessionId});
        }
        Preconditions.checkState((boolean)sessionBlocks.remove(blockId));
        if (sessionBlocks.isEmpty()) {
            this.mSessionIdToTempBlockIdsMap.remove(sessionId);
        }
        this.reclaimSpace(tempBlockMeta.getBlockSize(), false);
    }

    public void resizeTempBlockMeta(TempBlockMeta tempBlockMeta, long newSize) throws InvalidWorkerStateException {
        long oldSize = tempBlockMeta.getBlockSize();
        if (newSize > oldSize) {
            this.reserveSpace(newSize - oldSize, false);
            tempBlockMeta.setBlockSize(newSize);
        } else if (newSize < oldSize) {
            throw new InvalidWorkerStateException("Shrinking block, not supported!");
        }
    }

    public void cleanupSessionTempBlocks(long sessionId, List<Long> tempBlockIds) {
        Set<Long> sessionTempBlocks = this.mSessionIdToTempBlockIdsMap.get(sessionId);
        if (sessionTempBlocks == null) {
            return;
        }
        for (Long tempBlockId : tempBlockIds) {
            if (!this.mBlockIdToTempBlockMap.containsKey(tempBlockId)) continue;
            sessionTempBlocks.remove(tempBlockId);
            TempBlockMeta tempBlockMeta = this.mBlockIdToTempBlockMap.remove(tempBlockId);
            if (tempBlockMeta != null) {
                this.reclaimSpace(tempBlockMeta.getBlockSize(), false);
                continue;
            }
            LOG.error("Cannot find blockId {} when cleanup sessionId {}", (Object)tempBlockId, (Object)sessionId);
        }
        if (sessionTempBlocks.isEmpty()) {
            this.mSessionIdToTempBlockIdsMap.remove(sessionId);
        } else {
            LOG.warn("Blocks still owned by session {} after cleanup.", (Object)sessionId);
        }
    }

    public List<TempBlockMeta> getSessionTempBlocks(long sessionId) {
        Set<Long> sessionTempBlockIds = this.mSessionIdToTempBlockIdsMap.get(sessionId);
        if (sessionTempBlockIds == null || sessionTempBlockIds.isEmpty()) {
            return Collections.emptyList();
        }
        ArrayList<TempBlockMeta> sessionTempBlocks = new ArrayList<TempBlockMeta>();
        for (long blockId : sessionTempBlockIds) {
            sessionTempBlocks.add(this.mBlockIdToTempBlockMap.get(blockId));
        }
        return sessionTempBlocks;
    }

    public BlockStoreLocation toBlockStoreLocation() {
        return new BlockStoreLocation(this.mTier.getTierAlias(), this.mDirIndex);
    }

    private void reclaimSpace(long size, boolean committed) {
        Preconditions.checkState((this.mCapacityBytes >= this.mAvailableBytes.get() + size ? 1 : 0) != 0, (Object)"Available bytes should always be less than total capacity bytes");
        this.mAvailableBytes.addAndGet(size);
        if (committed) {
            this.mCommittedBytes.addAndGet(-size);
        }
    }

    private void reserveSpace(long size, boolean committed) {
        Preconditions.checkState((size <= this.mAvailableBytes.get() ? 1 : 0) != 0, (Object)"Available bytes should always be non-negative");
        this.mAvailableBytes.addAndGet(-size);
        if (committed) {
            this.mCommittedBytes.addAndGet(size);
        }
    }
}

