/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.Closeable;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.Options;
import org.apache.hadoop.fs.ParentNotDirectoryException;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathIsNotDirectoryException;
import org.apache.hadoop.fs.UnresolvedLinkException;
import org.apache.hadoop.fs.permission.AclEntry;
import org.apache.hadoop.fs.permission.AclStatus;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.fs.permission.PermissionStatus;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.protocol.AclException;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.DirectoryListing;
import org.apache.hadoop.hdfs.protocol.FSLimitException;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.HdfsFileStatus;
import org.apache.hadoop.hdfs.protocol.HdfsLocatedFileStatus;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.protocol.LocatedBlocks;
import org.apache.hadoop.hdfs.protocol.QuotaExceededException;
import org.apache.hadoop.hdfs.protocol.SnapshotAccessControlException;
import org.apache.hadoop.hdfs.protocol.SnapshotException;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfo;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoUnderConstruction;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManager;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeDescriptor;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeStorageInfo;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.namenode.AclFeature;
import org.apache.hadoop.hdfs.server.namenode.AclStorage;
import org.apache.hadoop.hdfs.server.namenode.AclTransformation;
import org.apache.hadoop.hdfs.server.namenode.CacheManager;
import org.apache.hadoop.hdfs.server.namenode.ContentSummaryComputationContext;
import org.apache.hadoop.hdfs.server.namenode.DirectoryWithQuotaFeature;
import org.apache.hadoop.hdfs.server.namenode.FSImage;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeDirectory;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.INodeMap;
import org.apache.hadoop.hdfs.server.namenode.INodeReference;
import org.apache.hadoop.hdfs.server.namenode.INodeSymlink;
import org.apache.hadoop.hdfs.server.namenode.INodeWithAdditionalFields;
import org.apache.hadoop.hdfs.server.namenode.INodesInPath;
import org.apache.hadoop.hdfs.server.namenode.NameCache;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.Quota;
import org.apache.hadoop.hdfs.server.namenode.snapshot.INodeDirectorySnapshottable;
import org.apache.hadoop.hdfs.server.namenode.snapshot.Snapshot;
import org.apache.hadoop.hdfs.util.ByteArray;
import org.apache.hadoop.hdfs.util.ChunkedArrayList;
import org.apache.hadoop.hdfs.util.ReadOnlyList;
import org.apache.hadoop.util.Time;

public class FSDirectory
implements Closeable {
    @VisibleForTesting
    static boolean CHECK_RESERVED_FILE_NAMES = true;
    public static final String DOT_RESERVED_STRING = ".reserved";
    public static final String DOT_RESERVED_PATH_PREFIX = "/.reserved";
    public static final byte[] DOT_RESERVED = DFSUtil.string2Bytes(".reserved");
    public static final String DOT_INODES_STRING = ".inodes";
    public static final byte[] DOT_INODES = DFSUtil.string2Bytes(".inodes");
    INodeDirectory rootDir;
    FSImage fsImage;
    private final FSNamesystem namesystem;
    private volatile boolean ready = false;
    private final int maxComponentLength;
    private final int maxDirItems;
    private final int lsLimit;
    private final int contentCountLimit;
    private final INodeMap inodeMap;
    private long yieldCount = 0L;
    private final ReentrantReadWriteLock dirLock = new ReentrantReadWriteLock(true);
    private final Condition cond = this.dirLock.writeLock().newCondition();
    private final NameCache<ByteArray> nameCache;

    private static INodeDirectorySnapshottable createRoot(FSNamesystem namesystem) {
        INodeDirectory r = new INodeDirectory(16385L, INodeDirectory.ROOT_NAME, namesystem.createFsOwnerPermissions(new FsPermission(493)), 0L);
        r.addDirectoryWithQuotaFeature(Long.MAX_VALUE, -1L);
        INodeDirectorySnapshottable s = new INodeDirectorySnapshottable(r);
        s.setSnapshotQuota(0);
        return s;
    }

    void readLock() {
        this.dirLock.readLock().lock();
    }

    void readUnlock() {
        this.dirLock.readLock().unlock();
    }

    void writeLock() {
        this.dirLock.writeLock().lock();
    }

    void writeUnlock() {
        this.dirLock.writeLock().unlock();
    }

    boolean hasWriteLock() {
        return this.dirLock.isWriteLockedByCurrentThread();
    }

    boolean hasReadLock() {
        return this.dirLock.getReadHoldCount() > 0;
    }

    public int getReadHoldCount() {
        return this.dirLock.getReadHoldCount();
    }

    public int getWriteHoldCount() {
        return this.dirLock.getWriteHoldCount();
    }

    FSDirectory(FSImage fsImage, FSNamesystem ns, Configuration conf) {
        this.rootDir = FSDirectory.createRoot(ns);
        this.inodeMap = INodeMap.newInstance(this.rootDir);
        this.fsImage = fsImage;
        int configuredLimit = conf.getInt("dfs.ls.limit", 1000);
        this.lsLimit = configuredLimit > 0 ? configuredLimit : 1000;
        this.contentCountLimit = conf.getInt("dfs.content-summary.limit", 0);
        this.maxComponentLength = conf.getInt("dfs.namenode.fs-limits.max-component-length", 255);
        this.maxDirItems = conf.getInt("dfs.namenode.fs-limits.max-directory-items", 0x100000);
        int MAX_DIR_ITEMS = 6400000;
        Preconditions.checkArgument((this.maxDirItems > 0 && this.maxDirItems <= 6400000 ? 1 : 0) != 0, (Object)"Cannot set dfs.namenode.fs-limits.max-directory-items to a value less than 0 or greater than 6400000");
        int threshold = conf.getInt("dfs.namenode.name.cache.threshold", 10);
        NameNode.LOG.info((Object)("Caching file names occuring more than " + threshold + " times"));
        this.nameCache = new NameCache(threshold);
        this.namesystem = ns;
    }

    private FSNamesystem getFSNamesystem() {
        return this.namesystem;
    }

    private BlockManager getBlockManager() {
        return this.getFSNamesystem().getBlockManager();
    }

    public INodeDirectory getRoot() {
        return this.rootDir;
    }

    void imageLoadComplete() {
        Preconditions.checkState((!this.ready ? 1 : 0) != 0, (Object)"FSDirectory already loaded");
        this.setReady();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void setReady() {
        if (this.ready) {
            return;
        }
        this.writeLock();
        try {
            this.setReady(true);
            this.nameCache.initialized();
            this.cond.signalAll();
        }
        finally {
            this.writeUnlock();
        }
    }

    @VisibleForTesting
    boolean isReady() {
        return this.ready;
    }

    protected void setReady(boolean flag) {
        this.ready = flag;
    }

    private void incrDeletedFileCount(long count) {
        if (this.getFSNamesystem() != null) {
            NameNode.getNameNodeMetrics().incrFilesDeleted(count);
        }
    }

    @Override
    public void close() throws IOException {
        this.fsImage.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void waitForReady() {
        if (!this.ready) {
            this.writeLock();
            try {
                while (!this.ready) {
                    try {
                        this.cond.await(5000L, TimeUnit.MILLISECONDS);
                    }
                    catch (InterruptedException interruptedException) {}
                }
            }
            finally {
                this.writeUnlock();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    INodeFile addFile(String path, PermissionStatus permissions, short replication, long preferredBlockSize, String clientName, String clientMachine, DatanodeDescriptor clientNode) throws FileAlreadyExistsException, QuotaExceededException, UnresolvedLinkException, SnapshotAccessControlException, AclException {
        this.waitForReady();
        long modTime = Time.now();
        Path parent = new Path(path).getParent();
        if (parent == null) {
            return null;
        }
        if (!this.mkdirs(parent.toString(), permissions, true, modTime)) {
            return null;
        }
        INodeFile newNode = new INodeFile(this.namesystem.allocateNewInodeId(), null, permissions, modTime, modTime, BlockInfo.EMPTY_ARRAY, replication, preferredBlockSize);
        newNode.toUnderConstruction(clientName, clientMachine, clientNode);
        boolean added = false;
        this.writeLock();
        try {
            added = this.addINode(path, newNode);
        }
        finally {
            this.writeUnlock();
        }
        if (!added) {
            NameNode.stateChangeLog.info((Object)("DIR* addFile: failed to add " + path));
            return null;
        }
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug((Object)("DIR* addFile: " + path + " is added"));
        }
        return newNode;
    }

    INodeFile unprotectedAddFile(long id, String path, PermissionStatus permissions, List<AclEntry> aclEntries, short replication, long modificationTime, long atime, long preferredBlockSize, boolean underConstruction, String clientName, String clientMachine) {
        block7: {
            INodeFile newNode;
            assert (this.hasWriteLock());
            if (underConstruction) {
                newNode = new INodeFile(id, null, permissions, modificationTime, modificationTime, BlockInfo.EMPTY_ARRAY, replication, preferredBlockSize);
                newNode.toUnderConstruction(clientName, clientMachine, null);
            } else {
                newNode = new INodeFile(id, null, permissions, modificationTime, atime, BlockInfo.EMPTY_ARRAY, replication, preferredBlockSize);
            }
            try {
                if (this.addINode(path, newNode)) {
                    if (aclEntries != null) {
                        AclStorage.updateINodeAcl(newNode, aclEntries, 0x7FFFFFFE);
                    }
                    return newNode;
                }
            }
            catch (IOException e) {
                if (!NameNode.stateChangeLog.isDebugEnabled()) break block7;
                NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.unprotectedAddFile: exception when add " + path + " to the file system"), (Throwable)e);
            }
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    BlockInfo addBlock(String path, INodesInPath inodesInPath, Block block, DatanodeStorageInfo[] targets) throws IOException {
        this.waitForReady();
        this.writeLock();
        try {
            INodeFile fileINode = inodesInPath.getLastINode().asFile();
            Preconditions.checkState((boolean)fileINode.isUnderConstruction());
            this.updateCount(inodesInPath, 0L, fileINode.getBlockDiskspace(), true);
            BlockInfoUnderConstruction blockInfo = new BlockInfoUnderConstruction(block, fileINode.getFileReplication(), HdfsServerConstants.BlockUCState.UNDER_CONSTRUCTION, targets);
            this.getBlockManager().addBlockCollection(blockInfo, fileINode);
            fileINode.addBlock(blockInfo);
            if (NameNode.stateChangeLog.isDebugEnabled()) {
                NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.addBlock: " + path + " with " + block + " block is added to the in-memory " + "file system"));
            }
            BlockInfoUnderConstruction blockInfoUnderConstruction = blockInfo;
            return blockInfoUnderConstruction;
        }
        finally {
            this.writeUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void persistBlocks(String path, INodeFile file, boolean logRetryCache) {
        Preconditions.checkArgument((boolean)file.isUnderConstruction());
        this.waitForReady();
        this.writeLock();
        try {
            this.fsImage.getEditLog().logUpdateBlocks(path, file, logRetryCache);
            if (NameNode.stateChangeLog.isDebugEnabled()) {
                NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.persistBlocks: " + path + " with " + file.getBlocks().length + " blocks is persisted to the file system"));
            }
        }
        finally {
            this.writeUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void persistNewBlock(String path, INodeFile file) {
        Preconditions.checkArgument((boolean)file.isUnderConstruction());
        this.waitForReady();
        this.writeLock();
        try {
            this.fsImage.getEditLog().logAddBlock(path, file);
        }
        finally {
            this.writeUnlock();
        }
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.persistNewBlock: " + path + " with new block " + file.getLastBlock().toString() + ", current total block count is " + file.getBlocks().length));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void closeFile(String path, INodeFile file) {
        this.waitForReady();
        this.writeLock();
        try {
            this.fsImage.getEditLog().logCloseFile(path, file);
            if (NameNode.stateChangeLog.isDebugEnabled()) {
                NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.closeFile: " + path + " with " + file.getBlocks().length + " blocks is persisted to the file system"));
            }
        }
        finally {
            this.writeUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean removeBlock(String path, INodeFile fileNode, Block block) throws IOException {
        Preconditions.checkArgument((boolean)fileNode.isUnderConstruction());
        this.waitForReady();
        this.writeLock();
        try {
            boolean bl = this.unprotectedRemoveBlock(path, fileNode, block);
            return bl;
        }
        finally {
            this.writeUnlock();
        }
    }

    boolean unprotectedRemoveBlock(String path, INodeFile fileNode, Block block) throws IOException {
        boolean removed = fileNode.removeLastBlock(block);
        if (!removed) {
            return false;
        }
        this.getBlockManager().removeBlockFromMap(block);
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.removeBlock: " + path + " with " + block + " block is removed from the file system"));
        }
        INodesInPath iip = this.rootDir.getINodesInPath4Write(path, true);
        this.updateCount(iip, 0L, -fileNode.getBlockDiskspace(), true);
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Deprecated
    boolean renameTo(String src, String dst, boolean logRetryCache) throws QuotaExceededException, UnresolvedLinkException, FileAlreadyExistsException, SnapshotAccessControlException, IOException {
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.renameTo: " + src + " to " + dst));
        }
        this.waitForReady();
        long now = Time.now();
        this.writeLock();
        try {
            if (!this.unprotectedRenameTo(src, dst, now)) {
                boolean bl = false;
                return bl;
            }
        }
        finally {
            this.writeUnlock();
        }
        this.fsImage.getEditLog().logRename(src, dst, now, logRetryCache);
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void renameTo(String src, String dst, boolean logRetryCache, Options.Rename ... options) throws FileAlreadyExistsException, FileNotFoundException, ParentNotDirectoryException, QuotaExceededException, UnresolvedLinkException, IOException {
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.renameTo: " + src + " to " + dst));
        }
        this.waitForReady();
        long now = Time.now();
        this.writeLock();
        try {
            if (this.unprotectedRenameTo(src, dst, now, options)) {
                this.incrDeletedFileCount(1L);
            }
        }
        finally {
            this.writeUnlock();
        }
        this.fsImage.getEditLog().logRename(src, dst, now, logRetryCache, options);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Deprecated
    boolean unprotectedRenameTo(String src, String dst, long timestamp) throws QuotaExceededException, UnresolvedLinkException, FileAlreadyExistsException, SnapshotAccessControlException, IOException {
        INodeReference.WithCount withCount;
        int srcRefDstSnapshot;
        assert (this.hasWriteLock());
        INodesInPath srcIIP = this.rootDir.getINodesInPath4Write(src, false);
        INode srcInode = srcIIP.getLastINode();
        if (srcInode == null) {
            NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: failed to rename " + src + " to " + dst + " because source does not exist"));
            return false;
        }
        if (srcIIP.getINodes().length == 1) {
            NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: failed to rename " + src + " to " + dst + " because source is the root"));
            return false;
        }
        ArrayList<INodeDirectorySnapshottable> snapshottableDirs = new ArrayList<INodeDirectorySnapshottable>();
        FSDirectory.checkSnapshot(srcInode, snapshottableDirs);
        if (this.isDir(dst)) {
            dst = dst + "/" + new Path(src).getName();
        }
        if (dst.equals(src)) {
            return true;
        }
        if (srcInode.isSymlink() && dst.equals(srcInode.asSymlink().getSymlinkString())) {
            throw new FileAlreadyExistsException("Cannot rename symlink " + src + " to its target " + dst);
        }
        if (dst.startsWith(src) && dst.charAt(src.length()) == '/') {
            NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: failed to rename " + src + " to " + dst + " because destination starts with src"));
            return false;
        }
        byte[][] dstComponents = INode.getPathComponents(dst);
        INodesInPath dstIIP = this.getExistingPathINodes(dstComponents);
        if (dstIIP.isSnapshot()) {
            throw new SnapshotAccessControlException("Modification on RO snapshot is disallowed");
        }
        if (dstIIP.getLastINode() != null) {
            NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: failed to rename " + src + " to " + dst + " because destination exists"));
            return false;
        }
        INode dstParent = dstIIP.getINode(-2);
        if (dstParent == null) {
            NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: failed to rename " + src + " to " + dst + " because destination's parent does not exist"));
            return false;
        }
        this.verifyFsLimitsForRename(srcIIP, dstIIP);
        this.verifyQuotaForRename(srcIIP.getINodes(), dstIIP.getINodes());
        boolean added = false;
        INode srcChild = srcIIP.getLastINode();
        byte[] srcChildName = srcChild.getLocalNameBytes();
        boolean isSrcInSnapshot = srcChild.isInLatestSnapshot(srcIIP.getLatestSnapshotId());
        boolean srcChildIsReference = srcChild.isReference();
        if (isSrcInSnapshot) {
            srcChild = srcChild.recordModification(srcIIP.getLatestSnapshotId());
            srcIIP.setLastINode(srcChild);
        }
        Quota.Counts oldSrcCounts = Quota.Counts.newInstance();
        int n = srcRefDstSnapshot = srcChildIsReference ? srcChild.asReference().getDstSnapshotId() : 0x7FFFFFFE;
        if (isSrcInSnapshot) {
            INodeReference.WithName withName = srcIIP.getINode(-2).asDirectory().replaceChild4ReferenceWithName(srcChild, srcIIP.getLatestSnapshotId());
            withCount = (INodeReference.WithCount)withName.getReferredINode();
            srcChild = withName;
            srcIIP.setLastINode(srcChild);
            withCount.getReferredINode().computeQuotaUsage(oldSrcCounts, true);
        } else {
            withCount = srcChildIsReference ? (INodeReference.WithCount)srcChild.asReference().getReferredINode() : null;
        }
        try {
            INode toDst;
            long removedSrc = this.removeLastINode(srcIIP);
            if (removedSrc == -1L) {
                NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: failed to rename " + src + " to " + dst + " because the source can not be removed"));
                boolean bl = false;
                return bl;
            }
            if (dstParent.getParent() == null) {
                dstIIP = this.getExistingPathINodes(dstComponents);
                dstParent = dstIIP.getINode(-2);
            }
            srcChild = srcIIP.getLastINode();
            byte[] dstChildName = dstIIP.getLastLocalName();
            if (withCount == null) {
                srcChild.setLocalName(dstChildName);
                toDst = srcChild;
            } else {
                withCount.getReferredINode().setLocalName(dstChildName);
                int dstSnapshotId = dstIIP.getLatestSnapshotId();
                INodeReference.DstReference ref = new INodeReference.DstReference(dstParent.asDirectory(), withCount, dstSnapshotId);
                toDst = ref;
            }
            added = this.addLastINodeNoQuotaCheck(dstIIP, toDst);
            if (added) {
                if (NameNode.stateChangeLog.isDebugEnabled()) {
                    NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.unprotectedRenameTo: " + src + " is renamed to " + dst));
                }
                INode srcParent = srcIIP.getINode(-2);
                srcParent.updateModificationTime(timestamp, srcIIP.getLatestSnapshotId());
                dstParent = dstIIP.getINode(-2);
                dstParent.updateModificationTime(timestamp, dstIIP.getLatestSnapshotId());
                this.getFSNamesystem().unprotectedChangeLease(src, dst);
                if (isSrcInSnapshot) {
                    Quota.Counts newSrcCounts = srcChild.computeQuotaUsage(Quota.Counts.newInstance(), false);
                    newSrcCounts.subtract(oldSrcCounts);
                    srcParent.addSpaceConsumed(newSrcCounts.get(Quota.NAMESPACE), newSrcCounts.get(Quota.DISKSPACE), false);
                }
                boolean bl = true;
                return bl;
            }
        }
        finally {
            if (!added) {
                INodeDirectory srcParent = srcIIP.getINode(-2).asDirectory();
                INode oldSrcChild = srcChild;
                if (withCount == null) {
                    srcChild.setLocalName(srcChildName);
                } else if (!srcChildIsReference) {
                    INode originalChild;
                    srcChild = originalChild = withCount.getReferredINode();
                    srcChild.setLocalName(srcChildName);
                } else {
                    withCount.removeReference(oldSrcChild.asReference());
                    INodeReference.DstReference originalRef = new INodeReference.DstReference(srcParent, withCount, srcRefDstSnapshot);
                    srcChild = originalRef;
                    withCount.getReferredINode().setLocalName(srcChildName);
                }
                if (isSrcInSnapshot) {
                    srcParent.undoRename4ScrParent(oldSrcChild.asReference(), srcChild);
                } else {
                    this.addLastINodeNoQuotaCheck(srcIIP, srcChild);
                }
            }
        }
        NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: failed to rename " + src + " to " + dst));
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean unprotectedRenameTo(String src, String dst, long timestamp, Options.Rename ... options) throws FileAlreadyExistsException, FileNotFoundException, ParentNotDirectoryException, QuotaExceededException, UnresolvedLinkException, IOException {
        INodeReference.WithCount withCount;
        INode dstParent;
        assert (this.hasWriteLock());
        boolean overwrite = false;
        if (null != options) {
            for (Options.Rename option : options) {
                if (option != Options.Rename.OVERWRITE) continue;
                overwrite = true;
            }
        }
        String error = null;
        INodesInPath srcIIP = this.rootDir.getINodesInPath4Write(src, false);
        INode srcInode = srcIIP.getLastINode();
        if (srcInode == null) {
            error = "rename source " + src + " is not found.";
            NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: " + error));
            throw new FileNotFoundException(error);
        }
        if (srcIIP.getINodes().length == 1) {
            error = "rename source cannot be the root";
            NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: " + error));
            throw new IOException(error);
        }
        FSDirectory.checkSnapshot(srcInode, null);
        if (dst.equals(src)) {
            throw new FileAlreadyExistsException("The source " + src + " and destination " + dst + " are the same");
        }
        if (srcInode.isSymlink() && dst.equals(srcInode.asSymlink().getSymlinkString())) {
            throw new FileAlreadyExistsException("Cannot rename symlink " + src + " to its target " + dst);
        }
        if (dst.startsWith(src) && dst.charAt(src.length()) == '/') {
            error = "Rename destination " + dst + " is a directory or file under source " + src;
            NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: " + error));
            throw new IOException(error);
        }
        INodesInPath dstIIP = this.rootDir.getINodesInPath4Write(dst, false);
        if (dstIIP.getINodes().length == 1) {
            error = "rename destination cannot be the root";
            NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: " + error));
            throw new IOException(error);
        }
        INode dstInode = dstIIP.getLastINode();
        ArrayList<INodeDirectorySnapshottable> snapshottableDirs = new ArrayList<INodeDirectorySnapshottable>();
        if (dstInode != null) {
            ReadOnlyList<INode> children;
            if (dstInode.isDirectory() != srcInode.isDirectory()) {
                error = "Source " + src + " and destination " + dst + " must both be directories";
                NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: " + error));
                throw new IOException(error);
            }
            if (!overwrite) {
                error = "rename destination " + dst + " already exists";
                NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: " + error));
                throw new FileAlreadyExistsException(error);
            }
            if (dstInode.isDirectory() && !(children = dstInode.asDirectory().getChildrenList(0x7FFFFFFE)).isEmpty()) {
                error = "rename destination directory is not empty: " + dst;
                NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: " + error));
                throw new IOException(error);
            }
            FSDirectory.checkSnapshot(dstInode, snapshottableDirs);
        }
        if ((dstParent = dstIIP.getINode(-2)) == null) {
            error = "rename destination parent " + dst + " not found.";
            NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: " + error));
            throw new FileNotFoundException(error);
        }
        if (!dstParent.isDirectory()) {
            error = "rename destination parent " + dst + " is a file.";
            NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: " + error));
            throw new ParentNotDirectoryException(error);
        }
        this.verifyFsLimitsForRename(srcIIP, dstIIP);
        this.verifyQuotaForRename(srcIIP.getINodes(), dstIIP.getINodes());
        INode srcChild = srcIIP.getLastINode();
        byte[] srcChildName = srcChild.getLocalNameBytes();
        boolean isSrcInSnapshot = srcChild.isInLatestSnapshot(srcIIP.getLatestSnapshotId());
        boolean srcChildIsReference = srcChild.isReference();
        if (isSrcInSnapshot) {
            srcChild = srcChild.recordModification(srcIIP.getLatestSnapshotId());
            srcIIP.setLastINode(srcChild);
        }
        int srcRefDstSnapshot = srcChildIsReference ? srcChild.asReference().getDstSnapshotId() : 0x7FFFFFFE;
        Quota.Counts oldSrcCounts = Quota.Counts.newInstance();
        if (isSrcInSnapshot) {
            INodeReference.WithName withName = srcIIP.getINode(-2).asDirectory().replaceChild4ReferenceWithName(srcChild, srcIIP.getLatestSnapshotId());
            withCount = (INodeReference.WithCount)withName.getReferredINode();
            srcChild = withName;
            srcIIP.setLastINode(srcChild);
            withCount.getReferredINode().computeQuotaUsage(oldSrcCounts, true);
        } else {
            withCount = srcChildIsReference ? (INodeReference.WithCount)srcChild.asReference().getReferredINode() : null;
        }
        boolean undoRemoveSrc = true;
        long removedSrc = this.removeLastINode(srcIIP);
        if (removedSrc == -1L) {
            error = "Failed to rename " + src + " to " + dst + " because the source can not be removed";
            NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: " + error));
            throw new IOException(error);
        }
        if (dstParent.getParent() == null) {
            dstIIP = this.rootDir.getINodesInPath4Write(dst, false);
        }
        boolean undoRemoveDst = false;
        INode removedDst = null;
        try {
            INode toDst;
            if (dstInode != null && this.removeLastINode(dstIIP) != -1L) {
                removedDst = dstIIP.getLastINode();
                undoRemoveDst = true;
            }
            srcChild = srcIIP.getLastINode();
            byte[] dstChildName = dstIIP.getLastLocalName();
            if (withCount == null) {
                srcChild.setLocalName(dstChildName);
                toDst = srcChild;
            } else {
                withCount.getReferredINode().setLocalName(dstChildName);
                int dstSnapshotId = dstIIP.getLatestSnapshotId();
                INodeReference.DstReference ref = new INodeReference.DstReference(dstIIP.getINode(-2).asDirectory(), withCount, dstSnapshotId);
                toDst = ref;
            }
            if (this.addLastINodeNoQuotaCheck(dstIIP, toDst)) {
                undoRemoveSrc = false;
                if (NameNode.stateChangeLog.isDebugEnabled()) {
                    NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.unprotectedRenameTo: " + src + " is renamed to " + dst));
                }
                INode srcParent = srcIIP.getINode(-2);
                srcParent.updateModificationTime(timestamp, srcIIP.getLatestSnapshotId());
                dstParent = dstIIP.getINode(-2);
                dstParent.updateModificationTime(timestamp, dstIIP.getLatestSnapshotId());
                this.getFSNamesystem().unprotectedChangeLease(src, dst);
                long filesDeleted = -1L;
                if (removedDst != null) {
                    undoRemoveDst = false;
                    INode.BlocksMapUpdateInfo collectedBlocks = new INode.BlocksMapUpdateInfo();
                    ChunkedArrayList<INode> removedINodes = new ChunkedArrayList<INode>();
                    filesDeleted = removedDst.cleanSubtree(0x7FFFFFFE, dstIIP.getLatestSnapshotId(), collectedBlocks, removedINodes, true).get(Quota.NAMESPACE);
                    this.getFSNamesystem().removePathAndBlocks(src, collectedBlocks, removedINodes);
                }
                if (snapshottableDirs.size() > 0) {
                    this.namesystem.removeSnapshottableDirs(snapshottableDirs);
                }
                if (isSrcInSnapshot) {
                    Quota.Counts newSrcCounts = srcChild.computeQuotaUsage(Quota.Counts.newInstance(), false);
                    newSrcCounts.subtract(oldSrcCounts);
                    srcParent.addSpaceConsumed(newSrcCounts.get(Quota.NAMESPACE), newSrcCounts.get(Quota.DISKSPACE), false);
                }
                boolean bl = filesDeleted >= 0L;
                return bl;
            }
        }
        finally {
            if (undoRemoveSrc) {
                INodeDirectory srcParent = srcIIP.getINode(-2).asDirectory();
                INode oldSrcChild = srcChild;
                if (withCount == null) {
                    srcChild.setLocalName(srcChildName);
                } else if (!srcChildIsReference) {
                    INode originalChild;
                    srcChild = originalChild = withCount.getReferredINode();
                    srcChild.setLocalName(srcChildName);
                } else {
                    withCount.removeReference(oldSrcChild.asReference());
                    INodeReference.DstReference originalRef = new INodeReference.DstReference(srcParent, withCount, srcRefDstSnapshot);
                    srcChild = originalRef;
                    withCount.getReferredINode().setLocalName(srcChildName);
                }
                if (srcParent.isWithSnapshot()) {
                    srcParent.undoRename4ScrParent(oldSrcChild.asReference(), srcChild);
                } else {
                    this.addLastINodeNoQuotaCheck(srcIIP, srcChild);
                }
            }
            if (undoRemoveDst) {
                if (dstParent.isDirectory() && dstParent.asDirectory().isWithSnapshot()) {
                    dstParent.asDirectory().undoRename4DstParent(removedDst, dstIIP.getLatestSnapshotId());
                } else {
                    this.addLastINodeNoQuotaCheck(dstIIP, removedDst);
                }
                if (removedDst.isReference()) {
                    INodeReference removedDstRef = removedDst.asReference();
                    INodeReference.WithCount wc = (INodeReference.WithCount)removedDstRef.getReferredINode().asReference();
                    wc.addReference(removedDstRef);
                }
            }
        }
        NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: failed to rename " + src + " to " + dst));
        throw new IOException("rename from " + src + " to " + dst + " failed.");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    Block[] setReplication(String src, short replication, short[] blockRepls) throws QuotaExceededException, UnresolvedLinkException, SnapshotAccessControlException {
        this.waitForReady();
        this.writeLock();
        try {
            Block[] fileBlocks = this.unprotectedSetReplication(src, replication, blockRepls);
            if (fileBlocks != null) {
                this.fsImage.getEditLog().logSetReplication(src, replication);
            }
            Block[] blockArray = fileBlocks;
            return blockArray;
        }
        finally {
            this.writeUnlock();
        }
    }

    Block[] unprotectedSetReplication(String src, short replication, short[] blockRepls) throws QuotaExceededException, UnresolvedLinkException, SnapshotAccessControlException {
        short newBR;
        assert (this.hasWriteLock());
        INodesInPath iip = this.rootDir.getINodesInPath4Write(src, true);
        INode inode = iip.getLastINode();
        if (inode == null || !inode.isFile()) {
            return null;
        }
        INodeFile file = inode.asFile();
        short oldBR = file.getBlockReplication();
        if (replication > oldBR) {
            long dsDelta = (long)(replication - oldBR) * (file.diskspaceConsumed() / (long)oldBR);
            this.updateCount(iip, 0L, dsDelta, true);
        }
        if ((newBR = (file = file.setFileReplication(replication, iip.getLatestSnapshotId(), this.inodeMap)).getBlockReplication()) < oldBR) {
            long dsDelta = (long)(newBR - oldBR) * (file.diskspaceConsumed() / (long)newBR);
            this.updateCount(iip, 0L, dsDelta, true);
        }
        if (blockRepls != null) {
            blockRepls[0] = oldBR;
            blockRepls[1] = newBR;
        }
        return file.getBlocks();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    long getPreferredBlockSize(String path) throws UnresolvedLinkException, FileNotFoundException, IOException {
        this.readLock();
        try {
            long l = INodeFile.valueOf(this.rootDir.getNode(path, false), path).getPreferredBlockSize();
            return l;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean exists(String src) throws UnresolvedLinkException {
        src = this.normalizePath(src);
        this.readLock();
        try {
            INode inode = this.rootDir.getNode(src, false);
            if (inode == null) {
                boolean bl = false;
                return bl;
            }
            boolean bl = !inode.isFile() || inode.asFile().getBlocks() != null;
            return bl;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void setPermission(String src, FsPermission permission) throws FileNotFoundException, UnresolvedLinkException, QuotaExceededException, SnapshotAccessControlException {
        this.writeLock();
        try {
            this.unprotectedSetPermission(src, permission);
        }
        finally {
            this.writeUnlock();
        }
        this.fsImage.getEditLog().logSetPermissions(src, permission);
    }

    void unprotectedSetPermission(String src, FsPermission permissions) throws FileNotFoundException, UnresolvedLinkException, QuotaExceededException, SnapshotAccessControlException {
        assert (this.hasWriteLock());
        INodesInPath inodesInPath = this.rootDir.getINodesInPath4Write(src, true);
        INode inode = inodesInPath.getLastINode();
        if (inode == null) {
            throw new FileNotFoundException("File does not exist: " + src);
        }
        int snapshotId = inodesInPath.getLatestSnapshotId();
        inode.setPermission(permissions, snapshotId);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void setOwner(String src, String username, String groupname) throws FileNotFoundException, UnresolvedLinkException, QuotaExceededException, SnapshotAccessControlException {
        this.writeLock();
        try {
            this.unprotectedSetOwner(src, username, groupname);
        }
        finally {
            this.writeUnlock();
        }
        this.fsImage.getEditLog().logSetOwner(src, username, groupname);
    }

    void unprotectedSetOwner(String src, String username, String groupname) throws FileNotFoundException, UnresolvedLinkException, QuotaExceededException, SnapshotAccessControlException {
        assert (this.hasWriteLock());
        INodesInPath inodesInPath = this.rootDir.getINodesInPath4Write(src, true);
        INode inode = inodesInPath.getLastINode();
        if (inode == null) {
            throw new FileNotFoundException("File does not exist: " + src);
        }
        if (username != null) {
            inode = inode.setUser(username, inodesInPath.getLatestSnapshotId());
        }
        if (groupname != null) {
            inode.setGroup(groupname, inodesInPath.getLatestSnapshotId());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void concat(String target, String[] srcs, boolean supportRetryCache) throws UnresolvedLinkException, QuotaExceededException, SnapshotAccessControlException, SnapshotException {
        this.writeLock();
        try {
            this.waitForReady();
            long timestamp = Time.now();
            this.unprotectedConcat(target, srcs, timestamp);
            this.fsImage.getEditLog().logConcat(target, srcs, timestamp, supportRetryCache);
        }
        finally {
            this.writeUnlock();
        }
    }

    void unprotectedConcat(String target, String[] srcs, long timestamp) throws UnresolvedLinkException, QuotaExceededException, SnapshotAccessControlException, SnapshotException {
        assert (this.hasWriteLock());
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug((Object)("DIR* FSNamesystem.concat to " + target));
        }
        INodesInPath trgIIP = this.rootDir.getINodesInPath4Write(target, true);
        INode[] trgINodes = trgIIP.getINodes();
        INodeFile trgInode = trgIIP.getLastINode().asFile();
        INodeDirectory trgParent = trgINodes[trgINodes.length - 2].asDirectory();
        int trgLatestSnapshot = trgIIP.getLatestSnapshotId();
        INodeFile[] allSrcInodes = new INodeFile[srcs.length];
        for (int i = 0; i < srcs.length; ++i) {
            INodesInPath iip = this.getINodesInPath4Write(srcs[i]);
            int latest = iip.getLatestSnapshotId();
            INode inode = iip.getLastINode();
            if (inode.isInLatestSnapshot(latest)) {
                throw new SnapshotException("Concat: the source file " + srcs[i] + " is in snapshot " + latest);
            }
            if (inode.isReference() && ((INodeReference.WithCount)inode.asReference().getReferredINode()).getReferenceCount() > 1) {
                throw new SnapshotException("Concat: the source file " + srcs[i] + " is referred by some other reference in some snapshot.");
            }
            allSrcInodes[i] = inode.asFile();
        }
        trgInode.concatBlocks(allSrcInodes);
        int count = 0;
        for (INodeFile nodeToRemove : allSrcInodes) {
            if (nodeToRemove == null) continue;
            nodeToRemove.setBlocks(null);
            trgParent.removeChild(nodeToRemove, trgLatestSnapshot);
            this.inodeMap.remove(nodeToRemove);
            ++count;
        }
        this.removeFromInodeMap(Arrays.asList(allSrcInodes));
        trgInode.setModificationTime(timestamp, trgLatestSnapshot);
        trgParent.updateModificationTime(timestamp, trgLatestSnapshot);
        FSDirectory.unprotectedUpdateCount(trgIIP, trgINodes.length - 1, -count, 0L);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean delete(String src, INode.BlocksMapUpdateInfo collectedBlocks, List<INode> removedINodes, boolean logRetryCache) throws IOException {
        long filesRemoved;
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.delete: " + src));
        }
        this.waitForReady();
        long now = Time.now();
        this.writeLock();
        try {
            INodesInPath inodesInPath = this.rootDir.getINodesInPath4Write(this.normalizePath(src), false);
            if (!FSDirectory.deleteAllowed(inodesInPath, src)) {
                filesRemoved = -1L;
            } else {
                ArrayList<INodeDirectorySnapshottable> snapshottableDirs = new ArrayList<INodeDirectorySnapshottable>();
                FSDirectory.checkSnapshot(inodesInPath.getLastINode(), snapshottableDirs);
                filesRemoved = this.unprotectedDelete(inodesInPath, collectedBlocks, removedINodes, now);
                this.namesystem.removeSnapshottableDirs(snapshottableDirs);
            }
        }
        finally {
            this.writeUnlock();
        }
        if (filesRemoved < 0L) {
            return false;
        }
        this.fsImage.getEditLog().logDelete(src, now, logRetryCache);
        this.incrDeletedFileCount(filesRemoved);
        this.getFSNamesystem().removePathAndBlocks(src, null, null);
        return true;
    }

    private static boolean deleteAllowed(INodesInPath iip, String src) {
        INode[] inodes = iip.getINodes();
        if (inodes == null || inodes.length == 0 || inodes[inodes.length - 1] == null) {
            if (NameNode.stateChangeLog.isDebugEnabled()) {
                NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.unprotectedDelete: failed to remove " + src + " because it does not exist"));
            }
            return false;
        }
        if (inodes.length == 1) {
            NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedDelete: failed to remove " + src + " because the root is not allowed to be deleted"));
            return false;
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean isNonEmptyDirectory(String path) throws UnresolvedLinkException {
        this.readLock();
        try {
            INodesInPath inodesInPath = this.rootDir.getLastINodeInPath(path, false);
            INode inode = inodesInPath.getINode(0);
            if (inode == null || !inode.isDirectory()) {
                boolean bl = false;
                return bl;
            }
            int s = inodesInPath.getPathSnapshotId();
            boolean bl = !inode.asDirectory().getChildrenList(s).isEmpty();
            return bl;
        }
        finally {
            this.readUnlock();
        }
    }

    void unprotectedDelete(String src, long mtime) throws UnresolvedLinkException, QuotaExceededException, SnapshotAccessControlException, IOException {
        assert (this.hasWriteLock());
        INode.BlocksMapUpdateInfo collectedBlocks = new INode.BlocksMapUpdateInfo();
        ChunkedArrayList<INode> removedINodes = new ChunkedArrayList<INode>();
        INodesInPath inodesInPath = this.rootDir.getINodesInPath4Write(this.normalizePath(src), false);
        long filesRemoved = -1L;
        if (FSDirectory.deleteAllowed(inodesInPath, src)) {
            ArrayList<INodeDirectorySnapshottable> snapshottableDirs = new ArrayList<INodeDirectorySnapshottable>();
            FSDirectory.checkSnapshot(inodesInPath.getLastINode(), snapshottableDirs);
            filesRemoved = this.unprotectedDelete(inodesInPath, collectedBlocks, removedINodes, mtime);
            this.namesystem.removeSnapshottableDirs(snapshottableDirs);
        }
        if (filesRemoved >= 0L) {
            this.getFSNamesystem().removePathAndBlocks(src, collectedBlocks, removedINodes);
        }
    }

    long unprotectedDelete(INodesInPath iip, INode.BlocksMapUpdateInfo collectedBlocks, List<INode> removedINodes, long mtime) throws QuotaExceededException {
        assert (this.hasWriteLock());
        INode targetNode = iip.getLastINode();
        if (targetNode == null) {
            return -1L;
        }
        int latestSnapshot = iip.getLatestSnapshotId();
        targetNode = targetNode.recordModification(latestSnapshot);
        iip.setLastINode(targetNode);
        long removed = this.removeLastINode(iip);
        if (removed == -1L) {
            return -1L;
        }
        INodeDirectory parent = targetNode.getParent();
        parent.updateModificationTime(mtime, latestSnapshot);
        if (removed == 0L) {
            return 0L;
        }
        if (!targetNode.isInLatestSnapshot(latestSnapshot)) {
            targetNode.destroyAndCollectBlocks(collectedBlocks, removedINodes);
        } else {
            Quota.Counts counts = targetNode.cleanSubtree(0x7FFFFFFE, latestSnapshot, collectedBlocks, removedINodes, true);
            parent.addSpaceConsumed(-counts.get(Quota.NAMESPACE), -counts.get(Quota.DISKSPACE), true);
            removed = counts.get(Quota.NAMESPACE);
        }
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.unprotectedDelete: " + targetNode.getFullPathName() + " is removed"));
        }
        return removed;
    }

    private static void checkSnapshot(INode target, List<INodeDirectorySnapshottable> snapshottableDirs) throws IOException {
        if (target.isDirectory()) {
            INodeDirectory targetDir = target.asDirectory();
            if (targetDir.isSnapshottable()) {
                INodeDirectorySnapshottable ssTargetDir = (INodeDirectorySnapshottable)targetDir;
                if (ssTargetDir.getNumSnapshots() > 0) {
                    throw new IOException("The directory " + ssTargetDir.getFullPathName() + " cannot be deleted since " + ssTargetDir.getFullPathName() + " is snapshottable and already has snapshots");
                }
                if (snapshottableDirs != null) {
                    snapshottableDirs.add(ssTargetDir);
                }
            }
            for (INode child : targetDir.getChildrenList(0x7FFFFFFE)) {
                FSDirectory.checkSnapshot(child, snapshottableDirs);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    DirectoryListing getListing(String src, byte[] startAfter, boolean needLocation) throws UnresolvedLinkException, IOException {
        String srcs = this.normalizePath(src);
        this.readLock();
        try {
            if (srcs.endsWith("/.snapshot")) {
                DirectoryListing directoryListing = this.getSnapshotsListing(srcs, startAfter);
                return directoryListing;
            }
            INodesInPath inodesInPath = this.rootDir.getLastINodeInPath(srcs, true);
            int snapshot = inodesInPath.getPathSnapshotId();
            INode targetNode = inodesInPath.getINode(0);
            if (targetNode == null) {
                DirectoryListing directoryListing = null;
                return directoryListing;
            }
            if (!targetNode.isDirectory()) {
                DirectoryListing directoryListing = new DirectoryListing(new HdfsFileStatus[]{this.createFileStatus(HdfsFileStatus.EMPTY_NAME, targetNode, needLocation, snapshot)}, 0);
                return directoryListing;
            }
            INodeDirectory dirInode = targetNode.asDirectory();
            ReadOnlyList<INode> contents = dirInode.getChildrenList(snapshot);
            int startChild = INodeDirectory.nextChild(contents, startAfter);
            int totalNumChildren = contents.size();
            int numOfListing = Math.min(totalNumChildren - startChild, this.lsLimit);
            int locationBudget = this.lsLimit;
            int listingCnt = 0;
            HdfsFileStatus[] listing = new HdfsFileStatus[numOfListing];
            for (int i = 0; i < numOfListing && locationBudget > 0; ++i) {
                INode cur = contents.get(startChild + i);
                listing[i] = this.createFileStatus(cur.getLocalNameBytes(), cur, needLocation, snapshot);
                ++listingCnt;
                if (!needLocation) continue;
                LocatedBlocks blks = ((HdfsLocatedFileStatus)listing[i]).getBlockLocations();
                locationBudget -= blks == null ? 0 : blks.locatedBlockCount() * listing[i].getReplication();
            }
            if (listingCnt < numOfListing) {
                listing = Arrays.copyOf(listing, listingCnt);
            }
            DirectoryListing directoryListing = new DirectoryListing(listing, totalNumChildren - startChild - listingCnt);
            return directoryListing;
        }
        finally {
            this.readUnlock();
        }
    }

    private DirectoryListing getSnapshotsListing(String src, byte[] startAfter) throws UnresolvedLinkException, IOException {
        Preconditions.checkState((boolean)this.hasReadLock());
        Preconditions.checkArgument((boolean)src.endsWith("/.snapshot"), (String)"%s does not end with %s", (Object[])new Object[]{src, "/.snapshot"});
        String dirPath = this.normalizePath(src.substring(0, src.length() - ".snapshot".length()));
        INode node = this.getINode(dirPath);
        INodeDirectorySnapshottable dirNode = INodeDirectorySnapshottable.valueOf(node, dirPath);
        ReadOnlyList<Snapshot> snapshots = dirNode.getSnapshotList();
        int skipSize = ReadOnlyList.Util.binarySearch(snapshots, startAfter);
        skipSize = skipSize < 0 ? -skipSize - 1 : skipSize + 1;
        int numOfListing = Math.min(snapshots.size() - skipSize, this.lsLimit);
        HdfsFileStatus[] listing = new HdfsFileStatus[numOfListing];
        for (int i = 0; i < numOfListing; ++i) {
            Snapshot.Root sRoot = snapshots.get(i + skipSize).getRoot();
            listing[i] = this.createFileStatus(sRoot.getLocalNameBytes(), sRoot, 0x7FFFFFFE);
        }
        return new DirectoryListing(listing, snapshots.size() - skipSize - numOfListing);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    HdfsFileStatus getFileInfo(String src, boolean resolveLink) throws UnresolvedLinkException {
        String srcs = this.normalizePath(src);
        this.readLock();
        try {
            if (srcs.endsWith("/.snapshot")) {
                HdfsFileStatus hdfsFileStatus = this.getFileInfo4DotSnapshot(srcs);
                return hdfsFileStatus;
            }
            INodesInPath inodesInPath = this.rootDir.getLastINodeInPath(srcs, resolveLink);
            INode i = inodesInPath.getINode(0);
            HdfsFileStatus hdfsFileStatus = i == null ? null : this.createFileStatus(HdfsFileStatus.EMPTY_NAME, i, inodesInPath.getPathSnapshotId());
            return hdfsFileStatus;
        }
        finally {
            this.readUnlock();
        }
    }

    private HdfsFileStatus getFileInfo4DotSnapshot(String src) throws UnresolvedLinkException {
        if (this.getINode4DotSnapshot(src) != null) {
            return new HdfsFileStatus(0L, true, 0, 0L, 0L, 0L, null, null, null, null, HdfsFileStatus.EMPTY_NAME, -1L, 0);
        }
        return null;
    }

    private INode getINode4DotSnapshot(String src) throws UnresolvedLinkException {
        Preconditions.checkArgument((boolean)src.endsWith("/.snapshot"), (String)"%s does not end with %s", (Object[])new Object[]{src, "/.snapshot"});
        String dirPath = this.normalizePath(src.substring(0, src.length() - ".snapshot".length()));
        INode node = this.getINode(dirPath);
        if (node != null && node.isDirectory() && node.asDirectory() instanceof INodeDirectorySnapshottable) {
            return node;
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    Block[] getFileBlocks(String src) throws UnresolvedLinkException {
        this.waitForReady();
        this.readLock();
        try {
            INode i = this.rootDir.getNode(src, false);
            Block[] blockArray = i != null && i.isFile() ? i.asFile().getBlocks() : null;
            return blockArray;
        }
        finally {
            this.readUnlock();
        }
    }

    INodesInPath getExistingPathINodes(byte[][] components) throws UnresolvedLinkException {
        return INodesInPath.resolve(this.rootDir, components);
    }

    public INode getINode(String src) throws UnresolvedLinkException {
        return this.getLastINodeInPath(src).getINode(0);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public INodesInPath getLastINodeInPath(String src) throws UnresolvedLinkException {
        this.readLock();
        try {
            INodesInPath iNodesInPath = this.rootDir.getLastINodeInPath(src, true);
            return iNodesInPath;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public INodesInPath getINodesInPath4Write(String src) throws UnresolvedLinkException, SnapshotAccessControlException {
        this.readLock();
        try {
            INodesInPath iNodesInPath = this.rootDir.getINodesInPath4Write(src, true);
            return iNodesInPath;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public INode getINode4Write(String src) throws UnresolvedLinkException, SnapshotAccessControlException {
        this.readLock();
        try {
            INode iNode = this.rootDir.getINode4Write(src, true);
            return iNode;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean isValidToCreate(String src) throws UnresolvedLinkException, SnapshotAccessControlException {
        String srcs = this.normalizePath(src);
        this.readLock();
        try {
            if (srcs.startsWith("/") && !srcs.endsWith("/") && this.rootDir.getINode4Write(srcs, false) == null) {
                boolean bl = true;
                return bl;
            }
            boolean bl = false;
            return bl;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean isDir(String src) throws UnresolvedLinkException {
        src = this.normalizePath(src);
        this.readLock();
        try {
            INode node = this.rootDir.getNode(src, false);
            boolean bl = node != null && node.isDirectory();
            return bl;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean isDirMutable(String src) throws UnresolvedLinkException, SnapshotAccessControlException {
        src = this.normalizePath(src);
        this.readLock();
        try {
            INode node = this.rootDir.getINode4Write(src, false);
            boolean bl = node != null && node.isDirectory();
            return bl;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void updateSpaceConsumed(String path, long nsDelta, long dsDelta) throws QuotaExceededException, FileNotFoundException, UnresolvedLinkException, SnapshotAccessControlException {
        this.writeLock();
        try {
            INodesInPath iip = this.rootDir.getINodesInPath4Write(path, false);
            if (iip.getLastINode() == null) {
                throw new FileNotFoundException("Path not found: " + path);
            }
            this.updateCount(iip, nsDelta, dsDelta, true);
        }
        finally {
            this.writeUnlock();
        }
    }

    private void updateCount(INodesInPath iip, long nsDelta, long dsDelta, boolean checkQuota) throws QuotaExceededException {
        this.updateCount(iip, iip.getINodes().length - 1, nsDelta, dsDelta, checkQuota);
    }

    private void updateCount(INodesInPath iip, int numOfINodes, long nsDelta, long dsDelta, boolean checkQuota) throws QuotaExceededException {
        assert (this.hasWriteLock());
        if (!this.ready) {
            return;
        }
        INode[] inodes = iip.getINodes();
        if (numOfINodes > inodes.length) {
            numOfINodes = inodes.length;
        }
        if (checkQuota) {
            FSDirectory.verifyQuota(inodes, numOfINodes, nsDelta, dsDelta, null);
        }
        FSDirectory.unprotectedUpdateCount(iip, numOfINodes, nsDelta, dsDelta);
    }

    private void updateCountNoQuotaCheck(INodesInPath inodesInPath, int numOfINodes, long nsDelta, long dsDelta) {
        assert (this.hasWriteLock());
        try {
            this.updateCount(inodesInPath, numOfINodes, nsDelta, dsDelta, false);
        }
        catch (QuotaExceededException e) {
            NameNode.LOG.error((Object)"BUG: unexpected exception ", (Throwable)e);
        }
    }

    private static void unprotectedUpdateCount(INodesInPath inodesInPath, int numOfINodes, long nsDelta, long dsDelta) {
        INode[] inodes = inodesInPath.getINodes();
        for (int i = 0; i < numOfINodes; ++i) {
            if (!inodes[i].isQuotaSet()) continue;
            inodes[i].asDirectory().getDirectoryWithQuotaFeature().addSpaceConsumed2Cache(nsDelta, dsDelta);
        }
    }

    static String getFullPathName(INode[] inodes, int pos) {
        StringBuilder fullPathName = new StringBuilder();
        if (inodes[0].isRoot()) {
            if (pos == 0) {
                return "/";
            }
        } else {
            fullPathName.append(inodes[0].getLocalName());
        }
        for (int i = 1; i <= pos; ++i) {
            fullPathName.append('/').append(inodes[i].getLocalName());
        }
        return fullPathName.toString();
    }

    private static INode[] getRelativePathINodes(INode inode, INode ancestor) {
        int depth = 0;
        for (INode i = inode; i != null && !i.equals(ancestor); i = i.getParent()) {
            ++depth;
        }
        INode[] inodes = new INode[depth];
        for (int i = 0; i < depth; ++i) {
            if (inode == null) {
                NameNode.stateChangeLog.warn((Object)"Could not get full path. Corresponding file might have deleted already.");
                return null;
            }
            inodes[depth - i - 1] = inode;
            inode = inode.getParent();
        }
        return inodes;
    }

    private static INode[] getFullPathINodes(INode inode) {
        return FSDirectory.getRelativePathINodes(inode, null);
    }

    static String getFullPathName(INode inode) {
        INode[] inodes = FSDirectory.getFullPathINodes(inode);
        return inodes == null ? "" : FSDirectory.getFullPathName(inodes, inodes.length - 1);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean mkdirs(String src, PermissionStatus permissions, boolean inheritPermission, long now) throws FileAlreadyExistsException, QuotaExceededException, UnresolvedLinkException, SnapshotAccessControlException, AclException {
        src = this.normalizePath(src);
        String[] names = INode.getPathNames(src);
        byte[][] components = INode.getPathComponents(names);
        int lastInodeIndex = components.length - 1;
        this.writeLock();
        try {
            int i;
            INodesInPath iip = this.getExistingPathINodes(components);
            if (iip.isSnapshot()) {
                throw new SnapshotAccessControlException("Modification on RO snapshot is disallowed");
            }
            INode[] inodes = iip.getINodes();
            StringBuilder pathbuilder = new StringBuilder();
            for (i = 1; i < inodes.length && inodes[i] != null; ++i) {
                pathbuilder.append("/").append(names[i]);
                if (inodes[i].isDirectory()) continue;
                throw new FileAlreadyExistsException("Parent path is not a directory: " + pathbuilder + " " + inodes[i].getLocalName());
            }
            PermissionStatus parentPermissions = permissions;
            if (inheritPermission || i < lastInodeIndex) {
                FsPermission parentFsPerm;
                FsPermission fsPermission = parentFsPerm = inheritPermission ? inodes[i - 1].getFsPermission() : permissions.getPermission();
                if (!parentFsPerm.getUserAction().implies(FsAction.WRITE_EXECUTE)) {
                    parentFsPerm = new FsPermission(parentFsPerm.getUserAction().or(FsAction.WRITE_EXECUTE), parentFsPerm.getGroupAction(), parentFsPerm.getOtherAction());
                }
                if (!parentPermissions.getPermission().equals((Object)parentFsPerm)) {
                    parentPermissions = new PermissionStatus(parentPermissions.getUserName(), parentPermissions.getGroupName(), parentFsPerm);
                    if (inheritPermission) {
                        permissions = parentPermissions;
                    }
                }
            }
            while (i < inodes.length) {
                pathbuilder.append("/" + names[i]);
                this.unprotectedMkdir(this.namesystem.allocateNewInodeId(), iip, i, components[i], i < lastInodeIndex ? parentPermissions : permissions, null, now);
                if (inodes[i] == null) {
                    boolean parentFsPerm = false;
                    return parentFsPerm;
                }
                if (this.getFSNamesystem() != null) {
                    NameNode.getNameNodeMetrics().incrFilesCreated();
                }
                String cur = pathbuilder.toString();
                this.fsImage.getEditLog().logMkDir(cur, inodes[i]);
                if (NameNode.stateChangeLog.isDebugEnabled()) {
                    NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.mkdirs: created directory " + cur));
                }
                ++i;
            }
        }
        finally {
            this.writeUnlock();
        }
        return true;
    }

    INode unprotectedMkdir(long inodeId, String src, PermissionStatus permissions, List<AclEntry> aclEntries, long timestamp) throws QuotaExceededException, UnresolvedLinkException, AclException {
        assert (this.hasWriteLock());
        byte[][] components = INode.getPathComponents(src);
        INodesInPath iip = this.getExistingPathINodes(components);
        INode[] inodes = iip.getINodes();
        int pos = inodes.length - 1;
        this.unprotectedMkdir(inodeId, iip, pos, components[pos], permissions, aclEntries, timestamp);
        return inodes[pos];
    }

    private void unprotectedMkdir(long inodeId, INodesInPath inodesInPath, int pos, byte[] name, PermissionStatus permission, List<AclEntry> aclEntries, long timestamp) throws QuotaExceededException, AclException {
        assert (this.hasWriteLock());
        INodeDirectory dir = new INodeDirectory(inodeId, name, permission, timestamp);
        if (this.addChild(inodesInPath, pos, dir, true)) {
            if (aclEntries != null) {
                AclStorage.updateINodeAcl(dir, aclEntries, 0x7FFFFFFE);
            }
            inodesInPath.setINode(pos, dir);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean addINode(String src, INode child) throws QuotaExceededException, UnresolvedLinkException {
        byte[][] components = INode.getPathComponents(src);
        child.setLocalName(components[components.length - 1]);
        this.cacheName(child);
        this.writeLock();
        try {
            boolean bl = this.addLastINode(this.getExistingPathINodes(components), child, true);
            return bl;
        }
        finally {
            this.writeUnlock();
        }
    }

    private static void verifyQuota(INode[] inodes, int pos, long nsDelta, long dsDelta, INode commonAncestor) throws QuotaExceededException {
        if (nsDelta <= 0L && dsDelta <= 0L) {
            return;
        }
        for (int i = (pos > inodes.length ? inodes.length : pos) - 1; i >= 0; --i) {
            if (commonAncestor == inodes[i]) {
                return;
            }
            DirectoryWithQuotaFeature q = inodes[i].asDirectory().getDirectoryWithQuotaFeature();
            if (q == null) continue;
            try {
                q.verifyQuota(nsDelta, dsDelta);
                continue;
            }
            catch (QuotaExceededException e) {
                e.setPathName(FSDirectory.getFullPathName(inodes, i));
                throw e;
            }
        }
    }

    private void verifyQuotaForRename(INode[] src, INode[] dst) throws QuotaExceededException {
        if (!this.ready) {
            return;
        }
        int i = 0;
        while (src[i] == dst[i]) {
            ++i;
        }
        Quota.Counts delta = src[src.length - 1].computeQuotaUsage();
        int dstIndex = dst.length - 1;
        if (dst[dstIndex] != null) {
            delta.subtract(dst[dstIndex].computeQuotaUsage());
        }
        FSDirectory.verifyQuota(dst, dstIndex, delta.get(Quota.NAMESPACE), delta.get(Quota.DISKSPACE), src[i - 1]);
    }

    private void verifyFsLimitsForRename(INodesInPath srcIIP, INodesInPath dstIIP) throws FSLimitException.PathComponentTooLongException, FSLimitException.MaxDirectoryItemsExceededException {
        byte[] dstChildName = dstIIP.getLastLocalName();
        INode[] dstInodes = dstIIP.getINodes();
        int pos = dstInodes.length - 1;
        this.verifyMaxComponentLength(dstChildName, dstInodes, pos);
        if (srcIIP.getINode(-2) != dstIIP.getINode(-2)) {
            this.verifyMaxDirItems(dstInodes, pos);
        }
    }

    void verifySnapshotName(String snapshotName, String path) throws FSLimitException.PathComponentTooLongException {
        if (snapshotName.contains("/")) {
            throw new HadoopIllegalArgumentException("Snapshot name cannot contain \"/\"");
        }
        byte[] bytes = DFSUtil.string2Bytes(snapshotName);
        this.verifyINodeName(bytes);
        this.verifyMaxComponentLength(bytes, path, 0);
    }

    void verifyINodeName(byte[] childName) throws HadoopIllegalArgumentException {
        if (Arrays.equals(HdfsConstants.DOT_SNAPSHOT_DIR_BYTES, childName)) {
            String s = "\".snapshot\" is a reserved name.";
            if (!this.ready) {
                s = s + "  Please rename it before upgrade.";
            }
            throw new HadoopIllegalArgumentException(s);
        }
    }

    private void verifyMaxComponentLength(byte[] childName, Object parentPath, int pos) throws FSLimitException.PathComponentTooLongException {
        if (this.maxComponentLength == 0) {
            return;
        }
        int length = childName.length;
        if (length > this.maxComponentLength) {
            String p = parentPath instanceof INode[] ? FSDirectory.getFullPathName((INode[])parentPath, pos - 1) : (String)parentPath;
            FSLimitException.PathComponentTooLongException e = new FSLimitException.PathComponentTooLongException(this.maxComponentLength, length, p, DFSUtil.bytes2String(childName));
            if (this.ready) {
                throw e;
            }
            NameNode.LOG.error((Object)"ERROR in FSDirectory.verifyINodeName", (Throwable)e);
        }
    }

    private void verifyMaxDirItems(INode[] pathComponents, int pos) throws FSLimitException.MaxDirectoryItemsExceededException {
        INodeDirectory parent = pathComponents[pos - 1].asDirectory();
        int count = parent.getChildrenList(0x7FFFFFFE).size();
        if (count >= this.maxDirItems) {
            FSLimitException.MaxDirectoryItemsExceededException e = new FSLimitException.MaxDirectoryItemsExceededException(this.maxDirItems, count);
            if (this.ready) {
                e.setPathName(FSDirectory.getFullPathName(pathComponents, pos - 1));
                throw e;
            }
            NameNode.LOG.error((Object)("FSDirectory.verifyMaxDirItems: " + e.getLocalizedMessage()));
        }
    }

    private boolean addLastINode(INodesInPath inodesInPath, INode inode, boolean checkQuota) throws QuotaExceededException {
        int pos = inodesInPath.getINodes().length - 1;
        return this.addChild(inodesInPath, pos, inode, checkQuota);
    }

    private boolean addChild(INodesInPath iip, int pos, INode child, boolean checkQuota) throws QuotaExceededException {
        INode[] inodes = iip.getINodes();
        if (pos == 1 && inodes[0] == this.rootDir && FSDirectory.isReservedName(child)) {
            throw new HadoopIllegalArgumentException("File name \"" + child.getLocalName() + "\" is reserved and cannot " + "be created. If this is during upgrade change the name of the " + "existing file or directory to another name before upgrading " + "to the new release.");
        }
        if (checkQuota) {
            this.verifyMaxComponentLength(child.getLocalNameBytes(), inodes, pos);
            this.verifyMaxDirItems(inodes, pos);
        }
        this.verifyINodeName(child.getLocalNameBytes());
        Quota.Counts counts = child.computeQuotaUsage();
        this.updateCount(iip, pos, counts.get(Quota.NAMESPACE), counts.get(Quota.DISKSPACE), checkQuota);
        boolean isRename = child.getParent() != null;
        INodeDirectory parent = inodes[pos - 1].asDirectory();
        boolean added = false;
        try {
            added = parent.addChild(child, true, iip.getLatestSnapshotId());
        }
        catch (QuotaExceededException e) {
            this.updateCountNoQuotaCheck(iip, pos, -counts.get(Quota.NAMESPACE), -counts.get(Quota.DISKSPACE));
            throw e;
        }
        if (!added) {
            this.updateCountNoQuotaCheck(iip, pos, -counts.get(Quota.NAMESPACE), -counts.get(Quota.DISKSPACE));
        } else {
            iip.setINode(pos - 1, child.getParent());
            if (!isRename) {
                AclStorage.copyINodeDefaultAcl(child);
            }
            this.addToInodeMap(child);
        }
        return added;
    }

    private boolean addLastINodeNoQuotaCheck(INodesInPath inodesInPath, INode i) {
        try {
            return this.addLastINode(inodesInPath, i, false);
        }
        catch (QuotaExceededException e) {
            NameNode.LOG.warn((Object)"FSDirectory.addChildNoQuotaCheck - unexpected", (Throwable)e);
            return false;
        }
    }

    private long removeLastINode(INodesInPath iip) throws QuotaExceededException {
        int latestSnapshot = iip.getLatestSnapshotId();
        INode last = iip.getLastINode();
        INodeDirectory parent = iip.getINode(-2).asDirectory();
        if (!parent.removeChild(last, latestSnapshot)) {
            return -1L;
        }
        INodeDirectory newParent = last.getParent();
        if (parent != newParent) {
            iip.setINode(-2, newParent);
        }
        if (!last.isInLatestSnapshot(latestSnapshot)) {
            Quota.Counts counts = last.computeQuotaUsage();
            this.updateCountNoQuotaCheck(iip, iip.getINodes().length - 1, -counts.get(Quota.NAMESPACE), -counts.get(Quota.DISKSPACE));
            if (INodeReference.tryRemoveReference(last) > 0) {
                return 0L;
            }
            return counts.get(Quota.NAMESPACE);
        }
        return 1L;
    }

    String normalizePath(String src) {
        if (src.length() > 1 && src.endsWith("/")) {
            src = src.substring(0, src.length() - 1);
        }
        return src;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    ContentSummary getContentSummary(String src) throws FileNotFoundException, UnresolvedLinkException {
        String srcs = this.normalizePath(src);
        this.readLock();
        try {
            INode targetNode = this.rootDir.getNode(srcs, false);
            if (targetNode == null) {
                throw new FileNotFoundException("File does not exist: " + srcs);
            }
            ContentSummaryComputationContext cscc = new ContentSummaryComputationContext(this, this.getFSNamesystem(), this.contentCountLimit);
            ContentSummary cs = targetNode.computeAndConvertContentSummary(cscc);
            this.yieldCount += cscc.getYieldCount();
            ContentSummary contentSummary = cs;
            return contentSummary;
        }
        finally {
            this.readUnlock();
        }
    }

    @VisibleForTesting
    public long getYieldCount() {
        return this.yieldCount;
    }

    public INodeMap getINodeMap() {
        return this.inodeMap;
    }

    public final void addToInodeMap(INode inode) {
        if (inode instanceof INodeWithAdditionalFields) {
            this.inodeMap.put((INodeWithAdditionalFields)inode);
        }
    }

    public final void removeFromInodeMap(List<? extends INode> inodes) {
        if (inodes != null) {
            for (INode iNode : inodes) {
                if (iNode == null || !(iNode instanceof INodeWithAdditionalFields)) continue;
                this.inodeMap.remove(iNode);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public INode getInode(long id) {
        this.readLock();
        try {
            INode iNode = this.inodeMap.get(id);
            return iNode;
        }
        finally {
            this.readUnlock();
        }
    }

    @VisibleForTesting
    int getInodeMapSize() {
        return this.inodeMap.size();
    }

    INodeDirectory unprotectedSetQuota(String src, long nsQuota, long dsQuota) throws FileNotFoundException, PathIsNotDirectoryException, QuotaExceededException, UnresolvedLinkException, SnapshotAccessControlException {
        assert (this.hasWriteLock());
        if (nsQuota < 0L && nsQuota != Long.MAX_VALUE && nsQuota < -1L || dsQuota < 0L && dsQuota != Long.MAX_VALUE && dsQuota < -1L) {
            throw new IllegalArgumentException("Illegal value for nsQuota or dsQuota : " + nsQuota + " and " + dsQuota);
        }
        String srcs = this.normalizePath(src);
        INodesInPath iip = this.rootDir.getINodesInPath4Write(srcs, true);
        INodeDirectory dirNode = INodeDirectory.valueOf(iip.getLastINode(), srcs);
        if (dirNode.isRoot() && nsQuota == -1L) {
            throw new IllegalArgumentException("Cannot clear namespace quota on root.");
        }
        Quota.Counts oldQuota = dirNode.getQuotaCounts();
        long oldNsQuota = oldQuota.get(Quota.NAMESPACE);
        long oldDsQuota = oldQuota.get(Quota.DISKSPACE);
        if (nsQuota == Long.MAX_VALUE) {
            nsQuota = oldNsQuota;
        }
        if (dsQuota == Long.MAX_VALUE) {
            dsQuota = oldDsQuota;
        }
        if (oldNsQuota == nsQuota && oldDsQuota == dsQuota) {
            return null;
        }
        int latest = iip.getLatestSnapshotId();
        dirNode = dirNode.recordModification(latest);
        dirNode.setQuota(nsQuota, dsQuota);
        return dirNode;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void setQuota(String src, long nsQuota, long dsQuota) throws FileNotFoundException, PathIsNotDirectoryException, QuotaExceededException, UnresolvedLinkException, SnapshotAccessControlException {
        this.writeLock();
        try {
            INodeDirectory dir = this.unprotectedSetQuota(src, nsQuota, dsQuota);
            if (dir != null) {
                Quota.Counts q = dir.getQuotaCounts();
                this.fsImage.getEditLog().logSetQuota(src, q.get(Quota.NAMESPACE), q.get(Quota.DISKSPACE));
            }
        }
        finally {
            this.writeUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    long totalInodes() {
        this.readLock();
        try {
            long l = this.rootDir.getDirectoryWithQuotaFeature().getSpaceConsumed().get(Quota.NAMESPACE);
            return l;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void setTimes(String src, INode inode, long mtime, long atime, boolean force, int latestSnapshotId) throws QuotaExceededException {
        boolean status = false;
        this.writeLock();
        try {
            status = this.unprotectedSetTimes(inode, mtime, atime, force, latestSnapshotId);
        }
        finally {
            this.writeUnlock();
        }
        if (status) {
            this.fsImage.getEditLog().logTimes(src, mtime, atime);
        }
    }

    boolean unprotectedSetTimes(String src, long mtime, long atime, boolean force) throws UnresolvedLinkException, QuotaExceededException {
        assert (this.hasWriteLock());
        INodesInPath i = this.getLastINodeInPath(src);
        return this.unprotectedSetTimes(i.getLastINode(), mtime, atime, force, i.getLatestSnapshotId());
    }

    private boolean unprotectedSetTimes(INode inode, long mtime, long atime, boolean force, int latest) throws QuotaExceededException {
        assert (this.hasWriteLock());
        boolean status = false;
        if (mtime != -1L) {
            inode = inode.setModificationTime(mtime, latest);
            status = true;
        }
        if (atime != -1L) {
            long inodeTime = inode.getAccessTime();
            if (atime <= inodeTime + this.getFSNamesystem().getAccessTimePrecision() && !force) {
                status = false;
            } else {
                inode.setAccessTime(atime, latest);
                status = true;
            }
        }
        return status;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void reset() {
        this.writeLock();
        try {
            this.setReady(false);
            this.rootDir = FSDirectory.createRoot(this.getFSNamesystem());
            this.inodeMap.clear();
            this.addToInodeMap(this.rootDir);
            this.nameCache.reset();
        }
        finally {
            this.writeUnlock();
        }
    }

    private HdfsFileStatus createFileStatus(byte[] path, INode node, boolean needLocation, int snapshot) throws IOException {
        if (needLocation) {
            return this.createLocatedFileStatus(path, node, snapshot);
        }
        return this.createFileStatus(path, node, snapshot);
    }

    HdfsFileStatus createFileStatus(byte[] path, INode node, int snapshot) {
        long size = 0L;
        short replication = 0;
        long blocksize = 0L;
        if (node.isFile()) {
            INodeFile fileNode = node.asFile();
            size = fileNode.computeFileSize(snapshot);
            replication = fileNode.getFileReplication(snapshot);
            blocksize = fileNode.getPreferredBlockSize();
        }
        int childrenNum = node.isDirectory() ? node.asDirectory().getChildrenNum(snapshot) : 0;
        return new HdfsFileStatus(size, node.isDirectory(), replication, blocksize, node.getModificationTime(snapshot), node.getAccessTime(snapshot), node.getFsPermission(snapshot), node.getUserName(snapshot), node.getGroupName(snapshot), node.isSymlink() ? node.asSymlink().getSymlink() : null, path, node.getId(), childrenNum);
    }

    private HdfsLocatedFileStatus createLocatedFileStatus(byte[] path, INode node, int snapshot) throws IOException {
        assert (this.hasReadLock());
        long size = 0L;
        short replication = 0;
        long blocksize = 0L;
        LocatedBlocks loc = null;
        if (node.isFile()) {
            INodeFile fileNode = node.asFile();
            size = fileNode.computeFileSize(snapshot);
            replication = fileNode.getFileReplication(snapshot);
            blocksize = fileNode.getPreferredBlockSize();
            boolean inSnapshot = snapshot != 0x7FFFFFFE;
            boolean isUc = inSnapshot ? false : fileNode.isUnderConstruction();
            long fileSize = !inSnapshot && isUc ? fileNode.computeFileSizeNotIncludingLastUcBlock() : size;
            loc = this.getFSNamesystem().getBlockManager().createLocatedBlocks(fileNode.getBlocks(), fileSize, isUc, 0L, size, false, inSnapshot);
            if (loc == null) {
                loc = new LocatedBlocks();
            }
        }
        int childrenNum = node.isDirectory() ? node.asDirectory().getChildrenNum(snapshot) : 0;
        HdfsLocatedFileStatus status = new HdfsLocatedFileStatus(size, node.isDirectory(), replication, blocksize, node.getModificationTime(snapshot), node.getAccessTime(snapshot), node.getFsPermission(snapshot), node.getUserName(snapshot), node.getGroupName(snapshot), node.isSymlink() ? node.asSymlink().getSymlink() : null, path, node.getId(), loc, childrenNum);
        if (loc != null) {
            CacheManager cacheManager = this.namesystem.getCacheManager();
            for (LocatedBlock lb : loc.getLocatedBlocks()) {
                cacheManager.setCachedLocations(lb);
            }
        }
        return status;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    INodeSymlink addSymlink(String path, String target, PermissionStatus dirPerms, boolean createParent, boolean logRetryCache) throws UnresolvedLinkException, FileAlreadyExistsException, QuotaExceededException, SnapshotAccessControlException, AclException {
        String parent;
        this.waitForReady();
        long modTime = Time.now();
        if (createParent && !this.mkdirs(parent = new Path(path).getParent().toString(), dirPerms, true, modTime)) {
            return null;
        }
        String userName = dirPerms.getUserName();
        INodeSymlink newNode = null;
        long id = this.namesystem.allocateNewInodeId();
        this.writeLock();
        try {
            newNode = this.unprotectedAddSymlink(id, path, target, modTime, modTime, new PermissionStatus(userName, null, FsPermission.getDefault()));
        }
        finally {
            this.writeUnlock();
        }
        if (newNode == null) {
            NameNode.stateChangeLog.info((Object)("DIR* addSymlink: failed to add " + path));
            return null;
        }
        this.fsImage.getEditLog().logSymlink(path, target, modTime, modTime, newNode, logRetryCache);
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug((Object)("DIR* addSymlink: " + path + " is added"));
        }
        return newNode;
    }

    INodeSymlink unprotectedAddSymlink(long id, String path, String target, long mtime, long atime, PermissionStatus perm) throws UnresolvedLinkException, QuotaExceededException {
        assert (this.hasWriteLock());
        INodeSymlink symlink = new INodeSymlink(id, null, perm, mtime, atime, target);
        return this.addINode(path, symlink) ? symlink : null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void modifyAclEntries(String src, List<AclEntry> aclSpec) throws IOException {
        this.writeLock();
        try {
            List<AclEntry> newAcl = this.unprotectedModifyAclEntries(src, aclSpec);
            this.fsImage.getEditLog().logSetAcl(src, newAcl);
        }
        finally {
            this.writeUnlock();
        }
    }

    private List<AclEntry> unprotectedModifyAclEntries(String src, List<AclEntry> aclSpec) throws IOException {
        assert (this.hasWriteLock());
        INodesInPath iip = this.rootDir.getINodesInPath4Write(this.normalizePath(src), true);
        INode inode = FSDirectory.resolveLastINode(src, iip);
        int snapshotId = iip.getLatestSnapshotId();
        List<AclEntry> existingAcl = AclStorage.readINodeLogicalAcl(inode);
        List<AclEntry> newAcl = AclTransformation.mergeAclEntries(existingAcl, aclSpec);
        AclStorage.updateINodeAcl(inode, newAcl, snapshotId);
        return newAcl;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void removeAclEntries(String src, List<AclEntry> aclSpec) throws IOException {
        this.writeLock();
        try {
            List<AclEntry> newAcl = this.unprotectedRemoveAclEntries(src, aclSpec);
            this.fsImage.getEditLog().logSetAcl(src, newAcl);
        }
        finally {
            this.writeUnlock();
        }
    }

    private List<AclEntry> unprotectedRemoveAclEntries(String src, List<AclEntry> aclSpec) throws IOException {
        assert (this.hasWriteLock());
        INodesInPath iip = this.rootDir.getINodesInPath4Write(this.normalizePath(src), true);
        INode inode = FSDirectory.resolveLastINode(src, iip);
        int snapshotId = iip.getLatestSnapshotId();
        List<AclEntry> existingAcl = AclStorage.readINodeLogicalAcl(inode);
        List<AclEntry> newAcl = AclTransformation.filterAclEntriesByAclSpec(existingAcl, aclSpec);
        AclStorage.updateINodeAcl(inode, newAcl, snapshotId);
        return newAcl;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void removeDefaultAcl(String src) throws IOException {
        this.writeLock();
        try {
            List<AclEntry> newAcl = this.unprotectedRemoveDefaultAcl(src);
            this.fsImage.getEditLog().logSetAcl(src, newAcl);
        }
        finally {
            this.writeUnlock();
        }
    }

    private List<AclEntry> unprotectedRemoveDefaultAcl(String src) throws IOException {
        assert (this.hasWriteLock());
        INodesInPath iip = this.rootDir.getINodesInPath4Write(this.normalizePath(src), true);
        INode inode = FSDirectory.resolveLastINode(src, iip);
        int snapshotId = iip.getLatestSnapshotId();
        List<AclEntry> existingAcl = AclStorage.readINodeLogicalAcl(inode);
        List<AclEntry> newAcl = AclTransformation.filterDefaultAclEntries(existingAcl);
        AclStorage.updateINodeAcl(inode, newAcl, snapshotId);
        return newAcl;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void removeAcl(String src) throws IOException {
        this.writeLock();
        try {
            this.unprotectedRemoveAcl(src);
            this.fsImage.getEditLog().logSetAcl(src, (List<AclEntry>)AclFeature.EMPTY_ENTRY_LIST);
        }
        finally {
            this.writeUnlock();
        }
    }

    private void unprotectedRemoveAcl(String src) throws IOException {
        assert (this.hasWriteLock());
        INodesInPath iip = this.rootDir.getINodesInPath4Write(this.normalizePath(src), true);
        INode inode = FSDirectory.resolveLastINode(src, iip);
        int snapshotId = iip.getLatestSnapshotId();
        AclStorage.removeINodeAcl(inode, snapshotId);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void setAcl(String src, List<AclEntry> aclSpec) throws IOException {
        this.writeLock();
        try {
            List<AclEntry> newAcl = this.unprotectedSetAcl(src, aclSpec);
            this.fsImage.getEditLog().logSetAcl(src, newAcl);
        }
        finally {
            this.writeUnlock();
        }
    }

    List<AclEntry> unprotectedSetAcl(String src, List<AclEntry> aclSpec) throws IOException {
        if (aclSpec.isEmpty()) {
            this.unprotectedRemoveAcl(src);
            return AclFeature.EMPTY_ENTRY_LIST;
        }
        assert (this.hasWriteLock());
        INodesInPath iip = this.rootDir.getINodesInPath4Write(this.normalizePath(src), true);
        INode inode = FSDirectory.resolveLastINode(src, iip);
        int snapshotId = iip.getLatestSnapshotId();
        List<AclEntry> existingAcl = AclStorage.readINodeLogicalAcl(inode);
        List<AclEntry> newAcl = AclTransformation.replaceAclEntries(existingAcl, aclSpec);
        AclStorage.updateINodeAcl(inode, newAcl, snapshotId);
        return newAcl;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    AclStatus getAclStatus(String src) throws IOException {
        String srcs = this.normalizePath(src);
        this.readLock();
        try {
            if (srcs.endsWith("/.snapshot") && this.getINode4DotSnapshot(srcs) != null) {
                AclStatus aclStatus = new AclStatus.Builder().owner("").group("").build();
                return aclStatus;
            }
            INodesInPath iip = this.rootDir.getLastINodeInPath(srcs, true);
            INode inode = FSDirectory.resolveLastINode(src, iip);
            int snapshotId = iip.getPathSnapshotId();
            List<AclEntry> acl = AclStorage.readINodeAcl(inode, snapshotId);
            AclStatus aclStatus = new AclStatus.Builder().owner(inode.getUserName()).group(inode.getGroupName()).stickyBit(inode.getFsPermission(snapshotId).getStickyBit()).addEntries(acl).build();
            return aclStatus;
        }
        finally {
            this.readUnlock();
        }
    }

    private static INode resolveLastINode(String src, INodesInPath iip) throws FileNotFoundException {
        INode inode = iip.getLastINode();
        if (inode == null) {
            throw new FileNotFoundException("cannot find " + src);
        }
        return inode;
    }

    void cacheName(INode inode) {
        if (!inode.isFile()) {
            return;
        }
        ByteArray name = new ByteArray(inode.getLocalNameBytes());
        if ((name = this.nameCache.put(name)) != null) {
            inode.setLocalName(name.getBytes());
        }
    }

    void shutdown() {
        this.nameCache.reset();
        this.inodeMap.clear();
    }

    static byte[][] getPathComponents(INode inode) {
        ArrayList<byte[]> components = new ArrayList<byte[]>();
        components.add(0, inode.getLocalNameBytes());
        while (inode.getParent() != null) {
            components.add(0, inode.getParent().getLocalNameBytes());
            inode = inode.getParent();
        }
        return (byte[][])components.toArray((T[])new byte[components.size()][]);
    }

    static byte[][] getPathComponentsForReservedPath(String src) {
        return !FSDirectory.isReservedName(src) ? (byte[][])null : INode.getPathComponents(src);
    }

    static String resolvePath(String src, byte[][] pathComponents, FSDirectory fsd) throws FileNotFoundException {
        if (pathComponents == null || pathComponents.length <= 3) {
            return src;
        }
        if (!Arrays.equals(DOT_RESERVED, pathComponents[1]) || !Arrays.equals(DOT_INODES, pathComponents[2])) {
            return src;
        }
        String inodeId = DFSUtil.bytes2String(pathComponents[3]);
        long id = 0L;
        try {
            id = Long.valueOf(inodeId);
        }
        catch (NumberFormatException e) {
            throw new FileNotFoundException("Invalid inode path: " + src);
        }
        if (id == 16385L && pathComponents.length == 4) {
            return "/";
        }
        INode inode = fsd.getInode(id);
        if (inode == null) {
            throw new FileNotFoundException("File for given inode path does not exist: " + src);
        }
        if (pathComponents.length > 4 && DFSUtil.bytes2String(pathComponents[4]).equals("..")) {
            INodeDirectory parent = inode.getParent();
            if (parent == null || ((INode)parent).getId() == 16385L) {
                return "/";
            }
            return parent.getFullPathName();
        }
        StringBuilder path = id == 16385L ? new StringBuilder() : new StringBuilder(inode.getFullPathName());
        for (int i = 4; i < pathComponents.length; ++i) {
            path.append("/").append(DFSUtil.bytes2String(pathComponents[i]));
        }
        if (NameNode.LOG.isDebugEnabled()) {
            NameNode.LOG.debug((Object)("Resolved path is " + path));
        }
        return path.toString();
    }

    public static boolean isReservedName(INode inode) {
        return CHECK_RESERVED_FILE_NAMES && Arrays.equals(inode.getLocalNameBytes(), DOT_RESERVED);
    }

    public static boolean isReservedName(String src) {
        return src.startsWith(DOT_RESERVED_PATH_PREFIX);
    }
}

