/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.ai.mcp.client.stdio;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.ai.mcp.client.stdio.ServerParameters;
import org.springframework.ai.mcp.spec.McpSchema;
import org.springframework.ai.mcp.spec.McpTransport;
import org.springframework.ai.mcp.util.Assert;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.publisher.Sinks;
import reactor.core.scheduler.Scheduler;
import reactor.core.scheduler.Schedulers;

public class StdioClientTransport
implements McpTransport {
    private static final Logger logger = LoggerFactory.getLogger(StdioClientTransport.class);
    private final Sinks.Many<McpSchema.JSONRPCMessage> inboundSink;
    private final Sinks.Many<McpSchema.JSONRPCMessage> outboundSink;
    private Process process;
    private ObjectMapper objectMapper;
    private Scheduler inboundScheduler;
    private Scheduler outboundScheduler;
    private Scheduler errorScheduler;
    private final ServerParameters params;
    private final Sinks.Many<String> errorSink;
    Consumer<String> errorHandler = error -> logger.error("Error received: {}", error);

    public StdioClientTransport(ServerParameters params) {
        this(params, new ObjectMapper());
    }

    public StdioClientTransport(ServerParameters params, ObjectMapper objectMapper) {
        Assert.notNull(params, "The params can not be null");
        Assert.notNull(objectMapper, "The ObjectMapper can not be null");
        this.inboundSink = Sinks.many().unicast().onBackpressureBuffer();
        this.outboundSink = Sinks.many().unicast().onBackpressureBuffer();
        this.params = params;
        this.objectMapper = objectMapper;
        this.errorSink = Sinks.many().unicast().onBackpressureBuffer();
        this.inboundScheduler = Schedulers.fromExecutorService((ExecutorService)Executors.newSingleThreadExecutor(), (String)"inbound");
        this.outboundScheduler = Schedulers.fromExecutorService((ExecutorService)Executors.newSingleThreadExecutor(), (String)"outbound");
        this.errorScheduler = Schedulers.fromExecutorService((ExecutorService)Executors.newSingleThreadExecutor(), (String)"error");
    }

    @Override
    public Mono<Void> connect(Function<Mono<McpSchema.JSONRPCMessage>, Mono<McpSchema.JSONRPCMessage>> handler) {
        return Mono.fromRunnable(() -> {
            this.handleIncomingMessages(handler);
            this.handleIncomingErrors();
            ArrayList<String> fullCommand = new ArrayList<String>();
            fullCommand.add(this.params.getCommand());
            fullCommand.addAll(this.params.getArgs());
            ProcessBuilder processBuilder = this.getProcessBuilder();
            processBuilder.command(fullCommand);
            processBuilder.environment().putAll(this.params.getEnv());
            try {
                this.process = processBuilder.start();
            }
            catch (IOException e) {
                throw new RuntimeException("Failed to start process with command: " + fullCommand, e);
            }
            if (this.process.getInputStream() == null || this.process.getOutputStream() == null) {
                this.process.destroy();
                throw new RuntimeException("Process input or output stream is null");
            }
            this.startInboundProcessing();
            this.startOutboundProcessing();
            this.startErrorProcessing();
        }).subscribeOn(Schedulers.boundedElastic());
    }

    protected ProcessBuilder getProcessBuilder() {
        return new ProcessBuilder(new String[0]);
    }

    public void setInboundErrorHandler(Consumer<String> errorHandler) {
        this.errorHandler = errorHandler;
    }

    public void awaitForExit() {
        try {
            this.process.waitFor();
        }
        catch (InterruptedException e) {
            throw new RuntimeException("Process interrupted", e);
        }
    }

    private void startErrorProcessing() {
        this.errorScheduler.schedule(() -> {
            try (BufferedReader processErrorReader = new BufferedReader(new InputStreamReader(this.process.getErrorStream()));){
                String line;
                while ((line = processErrorReader.readLine()) != null) {
                    try {
                        logger.error("Received error line: {}", (Object)line);
                        this.errorSink.tryEmitNext((Object)line);
                    }
                    catch (Exception e) {
                        throw new RuntimeException(e);
                        return;
                    }
                }
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        });
    }

    private void handleIncomingMessages(Function<Mono<McpSchema.JSONRPCMessage>, Mono<McpSchema.JSONRPCMessage>> inboundMessageHandler) {
        this.inboundSink.asFlux().flatMap(message -> Mono.just((Object)message).transform(inboundMessageHandler).contextWrite(ctx -> ctx.put((Object)"observation", (Object)"myObservation"))).subscribe();
    }

    private void handleIncomingErrors() {
        this.errorSink.asFlux().subscribe(e -> this.errorHandler.accept((String)e));
    }

    @Override
    public Mono<Void> sendMessage(McpSchema.JSONRPCMessage message) {
        if (this.outboundSink.tryEmitNext((Object)message).isSuccess()) {
            return Mono.empty();
        }
        return Mono.error((Throwable)new RuntimeException("Failed to enqueue message"));
    }

    private void startInboundProcessing() {
        this.inboundScheduler.schedule(() -> {
            try (BufferedReader processReader = new BufferedReader(new InputStreamReader(this.process.getInputStream()));){
                String line;
                while ((line = processReader.readLine()) != null) {
                    try {
                        McpSchema.JSONRPCMessage message = McpSchema.deserializeJsonRpcMessage(this.objectMapper, line);
                        if (this.inboundSink.tryEmitNext((Object)message).isSuccess()) continue;
                        throw new RuntimeException("Failed to enqueue message");
                    }
                    catch (Exception e) {
                        throw new RuntimeException(e);
                        return;
                    }
                }
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        });
    }

    private void startOutboundProcessing() {
        this.handleOutbound(messages -> messages.publishOn(this.outboundScheduler).handle((message, s) -> {
            if (message != null) {
                try {
                    String jsonMessage = this.objectMapper.writeValueAsString(message);
                    jsonMessage = jsonMessage.replace("\r\n", "\\n").replace("\n", "\\n").replace("\r", "\\n");
                    this.process.getOutputStream().write(jsonMessage.getBytes(StandardCharsets.UTF_8));
                    this.process.getOutputStream().write("\n".getBytes(StandardCharsets.UTF_8));
                    this.process.getOutputStream().flush();
                    s.next(message);
                }
                catch (IOException e) {
                    s.error((Throwable)new RuntimeException(e));
                }
            }
        }));
    }

    protected void handleOutbound(Function<Flux<McpSchema.JSONRPCMessage>, Flux<McpSchema.JSONRPCMessage>> outboundConsumer) {
        outboundConsumer.apply((Flux<McpSchema.JSONRPCMessage>)this.outboundSink.asFlux()).subscribe();
    }

    @Override
    public Mono<Void> closeGracefully() {
        return Mono.fromFuture(() -> {
            logger.info("Sending TERM to process");
            if (this.process != null) {
                this.process.destroy();
                return this.process.onExit();
            }
            return CompletableFuture.failedFuture(new RuntimeException("Process not started"));
        }).doOnNext(process -> {
            if (process.exitValue() != 0) {
                logger.warn("Process terminated with code " + process.exitValue());
            }
        }).then(Mono.fromRunnable(() -> {
            this.inboundScheduler.dispose();
            this.errorScheduler.dispose();
            this.outboundScheduler.dispose();
        })).then().subscribeOn(Schedulers.boundedElastic());
    }

    public Sinks.Many<String> getErrorSink() {
        return this.errorSink;
    }

    @Override
    public <T> T unmarshalFrom(Object data, TypeReference<T> typeRef) {
        return (T)this.objectMapper.convertValue(data, typeRef);
    }
}

