/*
 * 使用本软件请从杭州钛特云有限公司获取授权，其它途径获取本软件的行为皆为侵权行为
 * 本源代码由杭州钛特云科技有限公司授权给江苏南大先腾信息产业股份有限公司使用
 */
/**
 * Copyright (c) 2011-2019, James Zhan 詹波 (jfinal@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.tio.jfinal.template.ext.directive;

import org.tio.jfinal.template.Directive;
import org.tio.jfinal.template.Env;
import org.tio.jfinal.template.TemplateException;
import org.tio.jfinal.template.expr.ast.Expr;
import org.tio.jfinal.template.expr.ast.ExprList;
import org.tio.jfinal.template.io.Writer;
import org.tio.jfinal.template.stat.ParseException;
import org.tio.jfinal.template.stat.Scope;

import java.io.IOException;
import java.util.Date;

/**
 * #date 日期格式化输出指令
 *
 * 三种用法：
 * 1：#date(createAt) 用默认 datePattern 配置，输出 createAt 变量中的日期值
 * 2：#date(createAt, "yyyy-MM-dd HH:mm:ss") 用第二个参数指定的 datePattern，输出 createAt 变量中的日期值
 * 3：#date() 用默认 datePattern 配置，输出 “当前” 日期值
 *
 * 注意：
 * 1：#date 指令中的参数可以是变量，例如：#date(d, p) 中的 d 与 p 可以全都是变量
 * 2：默认 datePattern 可通过 Engine.setDatePattern(...) 进行配置
 */
public class DateDirective extends Directive {

	private Expr valueExpr;
	private Expr datePatternExpr;
	private int paraNum;

	public void setExprList(ExprList exprList) {
		this.paraNum = exprList.length();
		if (paraNum > 2) {
			throw new ParseException("Wrong number parameter of #date directive, two parameters allowed at most", location);
		}

		if (paraNum == 0) {
			this.valueExpr = null;
			this.datePatternExpr = null;
		} else if (paraNum == 1) {
			this.valueExpr = exprList.getExpr(0);
			this.datePatternExpr = null;
		} else if (paraNum == 2) {
			this.valueExpr = exprList.getExpr(0);
			this.datePatternExpr = exprList.getExpr(1);
		}
	}

	public void exec(Env env, Scope scope, Writer writer) {
		if (paraNum == 1) {
			outputWithoutDatePattern(env, scope, writer);
		} else if (paraNum == 2) {
			outputWithDatePattern(env, scope, writer);
		} else {
			outputToday(env, writer);
		}
	}

	private void outputToday(Env env, Writer writer) {
		write(writer, new Date(), env.getEngineConfig().getDatePattern());
	}

	private void outputWithoutDatePattern(Env env, Scope scope, Writer writer) {
		Object value = valueExpr.eval(scope);
		if (value instanceof Date) {
			write(writer, (Date)value, env.getEngineConfig().getDatePattern());
		} else if (value != null) {
			throw new TemplateException("The first parameter date of #date directive must be Date type", location);
		}
	}

	private void outputWithDatePattern(Env env, Scope scope, Writer writer) {
		Object value = valueExpr.eval(scope);
		if (value instanceof Date) {
			Object datePattern = this.datePatternExpr.eval(scope);
			if (datePattern instanceof String) {
				write(writer, (Date)value, (String)datePattern);
			} else {
				throw new TemplateException("The sencond parameter datePattern of #date directive must be String", location);
			}
		} else if (value != null) {
			throw new TemplateException("The first parameter date of #date directive must be Date type", location);
		}
	}

	private void write(Writer writer, Date date, String datePattern) {
		try {
			writer.write(date, datePattern);
		} catch (IOException e) {
			throw new TemplateException(e.getMessage(), location, e);
		}
	}
}








