package com.centit.framework.hibernate.service;

import java.io.Serializable;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import com.centit.framework.core.dao.PageDesc;
import com.centit.framework.core.service.BaseEntityManager;
import com.centit.framework.hibernate.dao.BaseDaoImpl;
import com.centit.support.algorithm.ReflectionOpt;

/**
 * 数据库的基本操作工具类
 * <p/>
 * 基本上是对Dao进行再一次简单的封装 注解Manager，添加默认事务
 *
 * @author codefan
 * @create 2012-2-16
 */
public abstract class BaseEntityManagerImpl<T extends Serializable, 
 		PK extends Serializable, D extends BaseDaoImpl<T, PK>> implements
        BaseEntityManager<T, PK> {

	/**
	 * 对应的Dao层对象
	 */
    protected D baseDao = null;

    protected void setBaseDao(D baseDao){
        this.baseDao = baseDao;
    }

    protected Log logger = LogFactory.getLog(BaseEntityManagerImpl.class);

    /**
     * 日志系统是否开启debug模式
     */
    protected boolean logDebug = logger.isDebugEnabled();

    @Transactional(propagation=Propagation.REQUIRED)
    public void deleteObject(T o) {
        baseDao.deleteObject(o);
    }

    @Transactional(propagation=Propagation.REQUIRED)
    public void deleteObjectById(PK id) {
        baseDao.deleteObjectById(id);
    }

    @Override
    @Transactional(propagation=Propagation.REQUIRED)
    public void saveObject(T o) {
        baseDao.saveObject(o);
    }

    /**
     * 保存泛型参数对象
     *
     * @param o
     */
    @Override
    @Transactional(propagation=Propagation.REQUIRED)
    public Serializable saveNewObject(T o){
        return baseDao.saveNewObject(o);
    }
    
    /**
     * 更新泛型参数对象
     *
     * @param o
     */
    @Override
    @Transactional(propagation=Propagation.REQUIRED)
    public void updateObject(T o){
        baseDao.saveObject(o); 
    }
    
    @Override
    @Transactional(propagation=Propagation.REQUIRED)
    public void mergeObject(T o) {
        baseDao.mergeObject(o);

    }
    
    
    /**
     * 修改之前check一下版本号，不一致抛异常
     */
    @Override
    @Transactional(propagation=Propagation.REQUIRED)
    public void updateObjectCheckTimestamp(T o) {
        baseDao.updateObjectCheckTimestamp(o);

    }
    
    @Override
    @Transactional(propagation=Propagation.REQUIRED)
    public T getObjectById(PK id) {
        return baseDao.getObjectById(id);
    }

    @Override
    @Transactional(propagation=Propagation.REQUIRED)
    public T getObjectByProperty(String propertyName, Object propertyValue) {
        return baseDao.getObjectByProperty(propertyName, propertyValue);
    }

    @Override
    @Transactional(propagation=Propagation.REQUIRED)
    public T getObjectByProperties(Map<String, Object> properties) {
        return baseDao.getObjectByProperties(properties);
    }

    public void copyObjectNotNullProperty(T des, T sou) {
        ReflectionOpt.invokeBinaryOpt(des, "copyNotNullProperty", sou);
    }

    public void clearObjectProperties(T des) {
        ReflectionOpt.invokeNoParamFunc(des, "clearProperties");

    }

    public void copyObject(T des, T sou) {
        ReflectionOpt.invokeBinaryOpt(des, "copy", sou);
    }
    
    @Override
    @Transactional(propagation=Propagation.REQUIRED)
    public List<T> listObjects() {
        return baseDao.listObjects();
    }

    @Override
    @Transactional(propagation=Propagation.REQUIRED)
    public List<T> listValidObjects() {
        return baseDao.listValidObjects();
    }

    /**
     * @param shql
     * @param filterMap 过滤条件
     * @return
     */
    @Override
    @Transactional(propagation=Propagation.REQUIRED)
    public List<T> listObjects(String shql, Map<String, Object> filterMap) {
        return baseDao.listObjects(shql, filterMap);
    }

    /**
     * @param filterMap 过滤条件
     * @return
     */
    @Override
    @Transactional(propagation=Propagation.REQUIRED)
    public List<T> listObjects(Map<String, Object> filterMap) {
        return baseDao.listObjects(filterMap,null);
    }
    
    /**
     * 查询数据库并且对查询结果分页
     */
    @Override
    @Transactional(propagation=Propagation.REQUIRED)
    public List<T> listObjects(String shql, Map<String, Object> filterMap, PageDesc pageDesc) {
        return baseDao.listObjects(shql, filterMap, pageDesc);
    }
    
    @Override
    @Transactional(propagation=Propagation.REQUIRED)
    public List<T> listObjects(Map<String, Object> filterMap, PageDesc pageDesc) {
        return baseDao.listObjects(filterMap, pageDesc);
    }

    /**
     * 获取数据库序列号
     */
/*    @Override
    @Transactional(propagation=Propagation.REQUIRED)
    public Long getNextLongSequence(String sequenceName) {
        return DatabaseOptUtils.getNextLongSequence(baseDao, sequenceName);
    }*/

}
