package com.aliyun.openservices.ons.sasl.client;

import javax.security.sasl.SaslClient;
import javax.security.sasl.SaslException;
import java.io.UnsupportedEncodingException;


public class OnsSaslClient implements SaslClient {

    public final static String ONS_MECHANISM = "ONS";
    private boolean completed = false;
    private String authorizationID;
    private String accessKey;
    private String secretKey;
    private String onsChannel = "ALIYUN";
    private final String VERSION = "0";
    private static byte SEP = 0;

    public OnsSaslClient(String authenticationID, String accessKey, String secretKey) throws SaslException {
        if (accessKey == null || secretKey == null) {
            throw new SaslException("ONS: AccessKey and SecretKey must be specified");
        }
        this.authorizationID = authenticationID;
        this.accessKey = accessKey;
        this.secretKey = secretKey;
    }

    public String getMechanismName() {
        return ONS_MECHANISM;
    }

    public boolean hasInitialResponse() {
        return false;
    }

    public void dispose() throws SaslException {
        this.clearPassword();
    }



    public byte[] evaluateChallenge(byte[] challenge) throws SaslException {
        if(this.completed) {
            throw new IllegalStateException("ONS authentication already completed");
        } else {
            if (challenge.length == 0) {
                this.completed = true;
                return null;
            }
            String[] tokens;
            try {
                tokens = new String(challenge, "UTF-8").split("\u0000");
            } catch (UnsupportedEncodingException var6) {
                throw new SaslException("Cannot get UTF-8 encoding of ids", var6);
            }

            if (tokens.length < 2) {
                throw new SaslException("Invalid SASL/ONS chan: expected equal or more than 2 tokens, got " + tokens.length);
            }
            String timestamp = tokens[0];
            String randomNumber = tokens[1];

            byte[] data = SaslUtil.join(SEP, VERSION.getBytes(), accessKey.getBytes(), timestamp.getBytes(), randomNumber.getBytes(), String.valueOf(ServiceAuthSigner.defaultAlgorithm).getBytes(), onsChannel.getBytes());
            String signaure = ServiceAuthSigner.calSignature(data, secretKey);

            byte[] response = SaslUtil.join(SEP, data, signaure.getBytes());
            return response;
        }
    }

    public boolean isComplete() {
        return this.completed;
    }

    public byte[] unwrap(byte[] incoming, int offset, int len) throws SaslException {
        if(this.completed) {
            throw new SaslException("PLAIN supports neither integrity nor privacy");
        } else {
            throw new IllegalStateException("PLAIN authentication not completed");
        }
    }

    public byte[] wrap(byte[] outgoing, int offset, int len) throws SaslException {
        if(this.completed) {
            throw new SaslException("PLAIN supports neither integrity nor privacy");
        } else {
            throw new IllegalStateException("PLAIN authentication not completed");
        }
    }

    public Object getNegotiatedProperty(String propName) {
        if(this.completed) {
            return propName.equals("javax.security.sasl.qop")?"auth":null;
        } else {
            throw new IllegalStateException("PLAIN authentication not completed");
        }
    }

    private void clearPassword() {
        if (this.secretKey != null) {
            this.secretKey = null;
        }
    }

    protected void finalize() {
        this.clearPassword();
    }
}